/*
 * Decompiled with CFR 0.152.
 */
package org.gradle.internal.watch.registry.impl;

import java.io.File;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Deque;
import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Stream;
import javax.annotation.CheckReturnValue;
import org.gradle.internal.file.FileHierarchySet;
import org.gradle.internal.file.FileMetadata;
import org.gradle.internal.snapshot.FileSystemLocationSnapshot;
import org.gradle.internal.snapshot.FileSystemSnapshotHierarchyVisitor;
import org.gradle.internal.snapshot.SnapshotHierarchy;
import org.gradle.internal.snapshot.SnapshotVisitResult;
import org.gradle.internal.watch.registry.FileWatcherProbeRegistry;
import org.gradle.internal.watch.registry.WatchMode;
import org.gradle.internal.watch.registry.impl.Combiners;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WatchableHierarchies {
    private static final Logger LOGGER = LoggerFactory.getLogger(WatchableHierarchies.class);
    public static final String INVALIDATING_HIERARCHY_MESSAGE = "Invalidating hierarchy because watch probe hasn't been triggered";
    private final FileWatcherProbeRegistry probeRegistry;
    private final Predicate<String> watchFilter;
    private FileHierarchySet watchableFiles = FileHierarchySet.empty();
    private FileHierarchySet unwatchableFiles = FileHierarchySet.empty();
    private final Deque<File> hierarchies = new ArrayDeque<File>();
    private final List<File> watchableHierarchiesSinceLastBuildFinish = new ArrayList<File>();

    public WatchableHierarchies(FileWatcherProbeRegistry probeRegistry, Predicate<String> watchFilter) {
        this.probeRegistry = probeRegistry;
        this.watchFilter = watchFilter;
    }

    public void registerWatchableHierarchy(File watchableHierarchy, SnapshotHierarchy root) {
        String watchableHierarchyPath = watchableHierarchy.getAbsolutePath();
        if (!this.watchFilter.test(watchableHierarchyPath)) {
            throw new IllegalStateException(String.format("Unable to watch directory '%s' since it is within Gradle's caches", watchableHierarchyPath));
        }
        this.watchableHierarchiesSinceLastBuildFinish.add(watchableHierarchy);
        if (this.unwatchableFiles.contains(watchableHierarchyPath)) {
            LOGGER.info("Not watching {} since the file system is not supported", (Object)watchableHierarchy);
            return;
        }
        this.doRegisterWatchableHierarchy(watchableHierarchy, root);
    }

    private void doRegisterWatchableHierarchy(File watchableHierarchy, SnapshotHierarchy root) {
        String watchableHierarchyPath = watchableHierarchy.getAbsolutePath();
        if (!this.watchableFiles.contains(watchableHierarchyPath)) {
            this.checkThatNothingExistsInNewWatchableHierarchy(watchableHierarchyPath, root);
            this.hierarchies.addFirst(watchableHierarchy);
            this.watchableFiles = this.watchableFiles.plus(watchableHierarchy);
        } else {
            this.hierarchies.remove(watchableHierarchy);
            this.hierarchies.addFirst(watchableHierarchy);
        }
        LOGGER.info("Now considering {} as hierarchies to watch", this.hierarchies);
    }

    @CheckReturnValue
    public SnapshotHierarchy removeUnwatchableContentOnBuildFinished(SnapshotHierarchy root, Predicate<File> isWatchedHierarchy, int maximumNumberOfWatchedHierarchies, List<File> unsupportedFileSystems, Invalidator invalidator) {
        SnapshotHierarchy newRoot = this.removeWatchedHierarchiesOverLimit(root, isWatchedHierarchy, maximumNumberOfWatchedHierarchies, invalidator);
        newRoot = this.removeUnwatchedSnapshots(newRoot, invalidator);
        if (!unsupportedFileSystems.isEmpty()) {
            newRoot = this.removeUnwatchableFileSystems(newRoot, unsupportedFileSystems, invalidator);
        }
        this.watchableHierarchiesSinceLastBuildFinish.clear();
        return newRoot;
    }

    @CheckReturnValue
    private SnapshotHierarchy removeUnwatchedSnapshots(SnapshotHierarchy root, Invalidator invalidator) {
        RemoveUnwatchedFiles removeUnwatchedFilesVisitor = new RemoveUnwatchedFiles(root, invalidator);
        root.rootSnapshots().forEach(snapshotRoot -> snapshotRoot.accept(removeUnwatchedFilesVisitor));
        return removeUnwatchedFilesVisitor.getRootWithUnwatchedFilesRemoved();
    }

    @CheckReturnValue
    private SnapshotHierarchy removeWatchedHierarchiesOverLimit(SnapshotHierarchy root, Predicate<File> isWatchedHierarchy, int maximumNumberOfWatchedHierarchies, Invalidator invalidator) {
        this.hierarchies.removeIf(hierarchy -> !isWatchedHierarchy.test((File)hierarchy));
        SnapshotHierarchy result2 = root;
        int toRemove = this.hierarchies.size() - maximumNumberOfWatchedHierarchies;
        if (toRemove > 0) {
            LOGGER.info("Watching too many directories in the file system (watching {}, limit {}), dropping some state from the virtual file system", (Object)this.hierarchies.size(), (Object)maximumNumberOfWatchedHierarchies);
            for (int i = 0; i < toRemove; ++i) {
                File locationToRemove = this.hierarchies.removeLast();
                result2 = invalidator.invalidate(locationToRemove.toString(), result2);
            }
        }
        this.watchableFiles = WatchableHierarchies.buildWatchableFilesFromHierarchies(this.hierarchies);
        return result2;
    }

    private static FileHierarchySet buildWatchableFilesFromHierarchies(Collection<File> hierarchies) {
        return hierarchies.stream().reduce(FileHierarchySet.empty(), FileHierarchySet::plus, Combiners.nonCombining());
    }

    @CheckReturnValue
    private SnapshotHierarchy removeUnwatchableFileSystems(SnapshotHierarchy root, List<File> unsupportedFileSystems, Invalidator invalidator) {
        SnapshotHierarchy invalidatedRoot = unsupportedFileSystems.stream().reduce(root, (updatedRoot, fileSystem) -> invalidator.invalidate(fileSystem.getAbsolutePath(), (SnapshotHierarchy)updatedRoot), Combiners.nonCombining());
        if (invalidatedRoot != root) {
            LOGGER.info("Some of the file system contents retained in the virtual file system are on file systems that Gradle doesn't support watching. The relevant state was discarded to ensure changes to these locations are properly detected. You can override this by explicitly enabling file system watching.");
        }
        return invalidatedRoot;
    }

    @CheckReturnValue
    public SnapshotHierarchy removeUnwatchableContentOnBuildStart(SnapshotHierarchy root, Invalidator invalidator, WatchMode watchMode, List<File> unsupportedFileSystems) {
        SnapshotHierarchy newRoot = root;
        newRoot = this.removeUnprovenHierarchies(newRoot, invalidator, watchMode);
        newRoot = this.updateUnwatchableFilesOnBuildStart(newRoot, invalidator, unsupportedFileSystems);
        return newRoot;
    }

    @CheckReturnValue
    private SnapshotHierarchy removeUnprovenHierarchies(SnapshotHierarchy root, Invalidator invalidator, WatchMode watchMode) {
        return this.probeRegistry.unprovenHierarchies().reduce(root, (currentRoot, unprovenHierarchy) -> {
            if (this.hierarchies.remove(unprovenHierarchy)) {
                watchMode.loggerForWarnings(LOGGER).warn("Invalidating hierarchy because watch probe hasn't been triggered {}", unprovenHierarchy);
                return invalidator.invalidate(unprovenHierarchy.getAbsolutePath(), (SnapshotHierarchy)currentRoot);
            }
            return currentRoot;
        }, Combiners.nonCombining());
    }

    @CheckReturnValue
    private SnapshotHierarchy updateUnwatchableFilesOnBuildStart(SnapshotHierarchy root, Invalidator invalidator, List<File> unsupportedFileSystems) {
        SnapshotHierarchy newRoot = root;
        FileHierarchySet oldUnwatchableFiles = this.unwatchableFiles;
        this.unwatchableFiles = unsupportedFileSystems.stream().reduce(FileHierarchySet.empty(), FileHierarchySet::plus, Combiners.nonCombining());
        if (!oldUnwatchableFiles.equals(this.unwatchableFiles)) {
            newRoot = this.invalidateUnwatchableHierarchies(newRoot, invalidator, oldUnwatchableFiles);
            newRoot = this.invalidateUnwatchableHierarchies(newRoot, invalidator, this.unwatchableFiles);
            this.hierarchies.removeIf(this.unwatchableFiles::contains);
            this.watchableFiles = WatchableHierarchies.buildWatchableFilesFromHierarchies(this.hierarchies);
            for (File watchableHierarchy : this.watchableHierarchiesSinceLastBuildFinish) {
                if (this.unwatchableFiles.contains(watchableHierarchy)) continue;
                this.doRegisterWatchableHierarchy(watchableHierarchy, newRoot);
            }
        }
        return newRoot;
    }

    @CheckReturnValue
    private SnapshotHierarchy invalidateUnwatchableHierarchies(SnapshotHierarchy root, Invalidator invalidator, FileHierarchySet unwatchableFiles) {
        InvalidatingRootVisitor invalidatingRootVisitor = new InvalidatingRootVisitor(root, invalidator);
        unwatchableFiles.visitRoots(invalidatingRootVisitor);
        return invalidatingRootVisitor.getNewRoot();
    }

    public Stream<File> stream() {
        return this.hierarchies.stream();
    }

    private void checkThatNothingExistsInNewWatchableHierarchy(String watchableHierarchy, SnapshotHierarchy vfsRoot) {
        vfsRoot.rootSnapshotsUnder(watchableHierarchy).filter(snapshotRoot -> !this.isInWatchableHierarchy(snapshotRoot.getAbsolutePath()) && !this.ignoredForWatching((FileSystemLocationSnapshot)snapshotRoot)).findAny().ifPresent(snapshotRoot -> {
            throw new IllegalStateException(String.format("Found existing snapshot at '%s' for unwatched hierarchy '%s'", snapshotRoot.getAbsolutePath(), watchableHierarchy));
        });
    }

    public boolean ignoredForWatching(FileSystemLocationSnapshot snapshot) {
        return snapshot.getAccessType() == FileMetadata.AccessType.VIA_SYMLINK || !this.watchFilter.test(snapshot.getAbsolutePath());
    }

    public boolean isInWatchableHierarchy(String path) {
        return this.watchableFiles.contains(path);
    }

    public boolean shouldWatch(FileSystemLocationSnapshot snapshot) {
        return !this.ignoredForWatching(snapshot) && this.isInWatchableHierarchy(snapshot.getAbsolutePath());
    }

    public static interface Invalidator {
        public SnapshotHierarchy invalidate(String var1, SnapshotHierarchy var2);
    }

    private class RemoveUnwatchedFiles
    implements FileSystemSnapshotHierarchyVisitor {
        private SnapshotHierarchy root;
        private final Invalidator invalidator;

        public RemoveUnwatchedFiles(SnapshotHierarchy root, Invalidator invalidator) {
            this.root = root;
            this.invalidator = invalidator;
        }

        @Override
        public SnapshotVisitResult visitEntry(FileSystemLocationSnapshot snapshot) {
            if (this.shouldBeRemoved(snapshot)) {
                this.invalidateUnwatchedFile(snapshot);
                return SnapshotVisitResult.SKIP_SUBTREE;
            }
            return SnapshotVisitResult.CONTINUE;
        }

        private boolean shouldBeRemoved(FileSystemLocationSnapshot snapshot) {
            return snapshot.getAccessType() == FileMetadata.AccessType.VIA_SYMLINK || !WatchableHierarchies.this.isInWatchableHierarchy(snapshot.getAbsolutePath()) && WatchableHierarchies.this.watchFilter.test(snapshot.getAbsolutePath());
        }

        private void invalidateUnwatchedFile(FileSystemLocationSnapshot snapshot) {
            this.root = this.invalidator.invalidate(snapshot.getAbsolutePath(), this.root);
        }

        public SnapshotHierarchy getRootWithUnwatchedFilesRemoved() {
            return this.root;
        }
    }

    private static class InvalidatingRootVisitor
    implements FileHierarchySet.RootVisitor {
        private SnapshotHierarchy newRoot;
        private final Invalidator invalidator;

        public InvalidatingRootVisitor(SnapshotHierarchy newRoot, Invalidator invalidator) {
            this.newRoot = newRoot;
            this.invalidator = invalidator;
        }

        @Override
        public void visitRoot(String absolutePath) {
            this.newRoot = this.invalidator.invalidate(absolutePath, this.newRoot);
        }

        public SnapshotHierarchy getNewRoot() {
            return this.newRoot;
        }
    }
}

