/*
 * Decompiled with CFR 0.152.
 */
package com.intellij.openapi.util.io;

import com.intellij.jna.JnaLoader;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.util.SystemInfo;
import com.intellij.openapi.util.io.FileAttributes;
import com.intellij.openapi.util.io.NioFiles;
import com.intellij.openapi.util.io.OSAgnosticPathUtil;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vfs.CharsetToolkit;
import com.intellij.util.SystemProperties;
import com.intellij.util.containers.LimitedPool;
import com.intellij.util.system.CpuArch;
import com.sun.jna.Library;
import com.sun.jna.Memory;
import com.sun.jna.Native;
import com.sun.jna.NativeLibrary;
import com.sun.jna.Platform;
import com.sun.jna.Pointer;
import com.sun.jna.Structure;
import com.sun.jna.platform.mac.CoreFoundation;
import com.sun.jna.platform.win32.Kernel32;
import com.sun.jna.platform.win32.WinBase;
import com.sun.jna.platform.win32.WinNT;
import com.sun.jna.ptr.LongByReference;
import com.sun.jna.ptr.PointerByReference;
import com.sun.jna.win32.StdCallLibrary;
import com.sun.jna.win32.W32APIOptions;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.DirectoryStream;
import java.nio.file.FileSystemException;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public final class FileSystemUtil {
    private static final Logger LOG = Logger.getInstance(FileSystemUtil.class);
    static final String FORCE_USE_NIO2_KEY = "idea.io.use.nio2";
    private static final String COARSE_TIMESTAMP_KEY = "idea.io.coarse.ts";
    @ApiStatus.Internal
    public static final boolean DO_NOT_RESOLVE_SYMLINKS = Boolean.getBoolean("idea.symlinks.no.resolve");
    private static volatile boolean MAC_CASE_SENSITIVITY_NATIVE_API_AVAILABLE = true;
    private static volatile boolean LINUX_CASE_SENSITIVITY_NATIVE_API_AVAILABLE = true;
    private static volatile boolean WINDOWS_CASE_SENSITIVITY_NATIVE_API_AVAILABLE = true;
    private static final Mediator ourMediator = FileSystemUtil.computeMediator();
    private static volatile boolean ourLibExt2FsPresent = true;

    @NotNull
    static Mediator computeMediator() {
        if (!Boolean.getBoolean(FORCE_USE_NIO2_KEY)) {
            try {
                if ((SystemInfo.isLinux || SystemInfo.isMac) && CpuArch.isIntel64() && JnaLoader.isLoaded()) {
                    return FileSystemUtil.ensureSane(new JnaUnixMediatorImpl());
                }
            }
            catch (Throwable t) {
                LOG.warn("Failed to load filesystem access layer: " + SystemInfo.OS_NAME + ", " + SystemInfo.JAVA_VERSION, t);
            }
        }
        return new Nio2MediatorImpl();
    }

    private static Mediator ensureSane(@NotNull Mediator mediator) throws Exception {
        String quickTestPath = SystemInfo.isWindows ? "C:\\" : "/";
        mediator.getAttributes(quickTestPath);
        return mediator;
    }

    private FileSystemUtil() {
    }

    @Nullable
    public static FileAttributes getAttributes(@NotNull String path) {
        try {
            if (LOG.isTraceEnabled()) {
                long t = System.nanoTime();
                FileAttributes result = ourMediator.getAttributes(path);
                t = System.nanoTime() - t;
                LOG.trace("getAttributes(" + path + ") = " + result + " in " + TimeUnit.NANOSECONDS.toMicros(t) + " mks");
                return result;
            }
            return ourMediator.getAttributes(path);
        }
        catch (Exception e) {
            LOG.warn("getAttributes(" + path + ")", e);
            return null;
        }
    }

    @Nullable
    public static FileAttributes getAttributes(@NotNull File file2) {
        return FileSystemUtil.getAttributes(file2.getPath());
    }

    public static long lastModified(@NotNull File file2) {
        FileAttributes attributes = FileSystemUtil.getAttributes(file2);
        return attributes != null ? attributes.lastModified : 0L;
    }

    public static boolean isSymLink(@NotNull String path) {
        FileAttributes attributes = FileSystemUtil.getAttributes(path);
        return attributes != null && attributes.isSymLink();
    }

    public static boolean isSymLink(@NotNull File file2) {
        return FileSystemUtil.isSymLink(file2.getAbsolutePath());
    }

    @Nullable
    public static String resolveSymLink(@NotNull String path) {
        try {
            String realPath;
            if (LOG.isTraceEnabled()) {
                long t = System.nanoTime();
                realPath = ourMediator.resolveSymLink(path);
                t = System.nanoTime() - t;
                LOG.trace("resolveSymLink(" + path + ") = " + realPath + " in " + TimeUnit.NANOSECONDS.toMicros(t) + " mks");
            } else {
                realPath = ourMediator.resolveSymLink(path);
            }
            if (realPath != null && (SystemInfo.isWindows && realPath.startsWith("\\\\") || new File(realPath).exists())) {
                return realPath;
            }
        }
        catch (Exception e) {
            LOG.warn(e);
        }
        return null;
    }

    @Nullable
    public static String resolveSymLink(@NotNull File file2) {
        return FileSystemUtil.resolveSymLink(file2.getAbsolutePath());
    }

    @NotNull
    public static FileAttributes.CaseSensitivity readParentCaseSensitivity(@NotNull File anyChild) {
        FileAttributes.CaseSensitivity detected = FileSystemUtil.readCaseSensitivityByNativeAPI(anyChild);
        if (detected != FileAttributes.CaseSensitivity.UNKNOWN) {
            return detected;
        }
        return FileSystemUtil.readParentCaseSensitivityByJavaIO(anyChild);
    }

    @NotNull
    static FileAttributes.CaseSensitivity readParentCaseSensitivityByJavaIO(@NotNull File anyChild) {
        String name;
        String altName;
        if (!anyChild.exists()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("readParentCaseSensitivityByJavaIO(" + anyChild + "): does not exist");
            }
            return FileAttributes.CaseSensitivity.UNKNOWN;
        }
        File parent = anyChild.getParentFile();
        if (parent == null) {
            String probe = FileSystemUtil.findCaseToggleableChild(anyChild);
            if (probe == null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("readParentCaseSensitivityByJavaIO(" + anyChild + "): no toggleable child, parent=null isDirectory=" + anyChild.isDirectory());
                }
                return FileAttributes.CaseSensitivity.UNKNOWN;
            }
            parent = anyChild;
            anyChild = new File(parent, probe);
        }
        if ((altName = FileSystemUtil.toggleCase(name = anyChild.getName())).equals(name)) {
            name = FileSystemUtil.findCaseToggleableChild(parent);
            if (name == null) {
                if (LOG.isDebugEnabled()) {
                    Object[] list = null;
                    try {
                        list = parent.list();
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                    if (list == null) {
                        LOG.debug("readParentCaseSensitivityByJavaIO(" + anyChild + "): parent.list() failed");
                    } else {
                        LOG.debug("readParentCaseSensitivityByJavaIO(" + anyChild + "): no toggleable child among " + list.length + " siblings");
                        if (LOG.isTraceEnabled()) {
                            LOG.trace("readParentCaseSensitivityByJavaIO(" + anyChild + "): " + Arrays.toString(list));
                        }
                    }
                }
                return FileAttributes.CaseSensitivity.UNKNOWN;
            }
            altName = FileSystemUtil.toggleCase(name);
        }
        String altPath = parent.getPath() + '/' + altName;
        FileAttributes newAttributes = FileSystemUtil.getAttributes(altPath);
        try {
            if (newAttributes == null) {
                return FileAttributes.CaseSensitivity.SENSITIVE;
            }
            File altCanonicalFile = new File(altPath).getCanonicalFile();
            String altCanonicalName = altCanonicalFile.getName();
            if (altCanonicalName.equals(name) || altCanonicalName.equals(anyChild.getCanonicalFile().getName())) {
                return FileAttributes.CaseSensitivity.INSENSITIVE;
            }
        }
        catch (IOException e) {
            LOG.debug("readParentCaseSensitivityByJavaIO(" + anyChild + ")", e);
            return FileAttributes.CaseSensitivity.UNKNOWN;
        }
        return FileAttributes.CaseSensitivity.SENSITIVE;
    }

    @NotNull
    static FileAttributes.CaseSensitivity readCaseSensitivityByNativeAPI(@NotNull File anyChild) {
        FileAttributes.CaseSensitivity detected = FileAttributes.CaseSensitivity.UNKNOWN;
        if (JnaLoader.isLoaded()) {
            File parent = anyChild.getParentFile();
            String path = (parent != null ? parent : anyChild).getAbsolutePath();
            if (SystemInfo.isWin10OrNewer && WINDOWS_CASE_SENSITIVITY_NATIVE_API_AVAILABLE) {
                detected = OSAgnosticPathUtil.isAbsoluteDosPath(path) ? FileSystemUtil.getNtfsCaseSensitivity(path) : FileAttributes.CaseSensitivity.UNKNOWN;
            } else if (SystemInfo.isMac && MAC_CASE_SENSITIVITY_NATIVE_API_AVAILABLE) {
                detected = FileSystemUtil.getMacOsCaseSensitivity(path);
            } else if (SystemInfo.isLinux && LINUX_CASE_SENSITIVITY_NATIVE_API_AVAILABLE) {
                detected = FileSystemUtil.getLinuxCaseSensitivity(path);
            }
        }
        return detected;
    }

    @NotNull
    private static String toggleCase(@NotNull String name) {
        String altName = name.toUpperCase(Locale.getDefault());
        if (altName.equals(name)) {
            altName = name.toLowerCase(Locale.getDefault());
        }
        return altName;
    }

    public static boolean isCaseToggleable(@NotNull String name) {
        return !FileSystemUtil.toggleCase(name).equals(name);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Nullable
    private static String findCaseToggleableChild(@NotNull File dir) {
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(dir.toPath());){
            Path path;
            String name;
            Iterator<Path> iterator = stream.iterator();
            do {
                if (!iterator.hasNext()) return null;
            } while ((name = (path = iterator.next()).getFileName().toString()).toLowerCase(Locale.getDefault()).equals(name.toUpperCase(Locale.getDefault())));
            String string2 = name;
            return string2;
        }
        catch (Exception exception) {
            // empty catch block
        }
        return null;
    }

    private static FileAttributes.CaseSensitivity getNtfsCaseSensitivity(String path) {
        NtOsKrnl ntOsKrnl;
        Kernel32 kernel32;
        try {
            kernel32 = Kernel32.INSTANCE;
            ntOsKrnl = NtOsKrnl.INSTANCE;
        }
        catch (Throwable e) {
            WINDOWS_CASE_SENSITIVITY_NATIVE_API_AVAILABLE = false;
            return FileAttributes.CaseSensitivity.UNKNOWN;
        }
        try {
            String name = "\\\\?\\" + path;
            WinNT.HANDLE handle = kernel32.CreateFile(name, 0, 7, null, 3, 0x2000000, null);
            if (handle == WinBase.INVALID_HANDLE_VALUE) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("CreateFile(" + path + "): 0x" + Integer.toHexString(kernel32.GetLastError()));
                }
                return FileAttributes.CaseSensitivity.UNKNOWN;
            }
            NtOsKrnl.FILE_CASE_SENSITIVE_INFORMATION_P fileInformation = new NtOsKrnl.FILE_CASE_SENSITIVE_INFORMATION_P();
            int result = ntOsKrnl.NtQueryInformationFile(handle, new NtOsKrnl.IO_STATUS_BLOCK_P(), fileInformation, fileInformation.size(), 71);
            kernel32.CloseHandle(handle);
            if (result != 0) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("NtQueryInformationFile(" + path + "): 0x" + Integer.toHexString(result));
                }
            } else {
                if (fileInformation.Flags == 0L) {
                    return FileAttributes.CaseSensitivity.INSENSITIVE;
                }
                if (fileInformation.Flags == 1L) {
                    return FileAttributes.CaseSensitivity.SENSITIVE;
                }
                LOG.warn("NtQueryInformationFile(" + path + "): unexpected 'FileCaseSensitiveInformation' value " + fileInformation.Flags);
            }
        }
        catch (Throwable t) {
            LOG.warn("path: " + path, t);
        }
        return FileAttributes.CaseSensitivity.UNKNOWN;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static FileAttributes.CaseSensitivity getMacOsCaseSensitivity(String path) {
        CoreFoundation cf;
        try {
            cf = CoreFoundation.INSTANCE;
        }
        catch (Throwable e) {
            MAC_CASE_SENSITIVITY_NATIVE_API_AVAILABLE = false;
            return FileAttributes.CaseSensitivity.UNKNOWN;
        }
        try {
            byte[] buffer = path.getBytes(StandardCharsets.UTF_8);
            CoreFoundation.CFTypeRef url = cf.CFURLCreateFromFileSystemRepresentation(null, buffer, buffer.length, true);
            try {
                PointerByReference resultPtr = new PointerByReference();
                PointerByReference errorPtr = new PointerByReference();
                if (!cf.CFURLCopyResourcePropertyForKey(url, CoreFoundation.kCFURLVolumeSupportsCaseSensitiveNamesKey, resultPtr, errorPtr)) {
                    if (!LOG.isDebugEnabled()) return FileAttributes.CaseSensitivity.UNKNOWN;
                    Pointer error = errorPtr.getValue();
                    String description = error != null ? cf.CFErrorGetDomain(error).stringValue() + '/' + cf.CFErrorGetCode(error) : "error";
                    LOG.debug("CFURLCopyResourcePropertyForKey(" + path + "): " + description);
                    return FileAttributes.CaseSensitivity.UNKNOWN;
                }
                Pointer result = resultPtr.getValue();
                if (result == null) {
                    if (!LOG.isDebugEnabled()) return FileAttributes.CaseSensitivity.UNKNOWN;
                    LOG.debug("CFURLCopyResourcePropertyForKey(" + path + "): property not available");
                    return FileAttributes.CaseSensitivity.UNKNOWN;
                }
                boolean value = new CoreFoundation.CFBooleanRef(result).booleanValue();
                FileAttributes.CaseSensitivity caseSensitivity = value ? FileAttributes.CaseSensitivity.SENSITIVE : FileAttributes.CaseSensitivity.INSENSITIVE;
                return caseSensitivity;
            }
            finally {
                url.release();
            }
        }
        catch (Throwable t) {
            LOG.warn("path: " + path, t);
        }
        return FileAttributes.CaseSensitivity.UNKNOWN;
    }

    private static FileAttributes.CaseSensitivity getLinuxCaseSensitivity(String path) {
        block13: {
            LibC libC;
            try {
                libC = LibC.INSTANCE;
            }
            catch (Throwable e) {
                ourLibExt2FsPresent = false;
                LINUX_CASE_SENSITIVITY_NATIVE_API_AVAILABLE = false;
                return FileAttributes.CaseSensitivity.UNKNOWN;
            }
            try {
                E2P e2P;
                long fs;
                Memory buf = new Memory(256L);
                if (libC.statfs(path, buf) != 0) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("statfs(" + path + "): error");
                    }
                    break block13;
                }
                long l = fs = Native.LONG_SIZE == 4 ? (long)buf.getInt(0L) & 0xFFFFFFFFL : buf.getLong(0L);
                if (fs == 2435016766L || fs == 1481003842L) {
                    return FileAttributes.CaseSensitivity.SENSITIVE;
                }
                if (fs == 19780L) {
                    return FileAttributes.CaseSensitivity.INSENSITIVE;
                }
                if (fs != 61267L && fs != 4076150800L || !ourLibExt2FsPresent) break block13;
                try {
                    e2P = E2P.INSTANCE;
                }
                catch (Throwable e) {
                    LINUX_CASE_SENSITIVITY_NATIVE_API_AVAILABLE = false;
                    return FileAttributes.CaseSensitivity.UNKNOWN;
                }
                LongByReference flags = new LongByReference();
                if (e2P.fgetflags(path, flags) != 0) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("fgetflags(" + path + "): error");
                    }
                    break block13;
                }
                return (flags.getValue() & 0x40000000L) == 0L ? FileAttributes.CaseSensitivity.SENSITIVE : FileAttributes.CaseSensitivity.INSENSITIVE;
            }
            catch (UnsatisfiedLinkError e) {
                ourLibExt2FsPresent = false;
                LOG.warn(e);
            }
            catch (Throwable t) {
                LOG.warn("path: " + path, t);
            }
        }
        return FileAttributes.CaseSensitivity.UNKNOWN;
    }

    private static class JnaUnixMediatorImpl
    implements Mediator {
        private static final int[] LINUX_64 = new int[]{24, 48, 88, 28, 32};
        private static final int[] BSD_64 = new int[]{8, 72, 40, 12, 16};
        private static final int STAT_VER = 1;
        private static final int OFF_MODE = 0;
        private static final int OFF_SIZE = 1;
        private static final int OFF_TIME = 2;
        private static final int OFF_UID = 3;
        private static final int OFF_GID = 4;
        private final int[] myOffsets;
        private final int myUid;
        private final int myGid;
        private final boolean myCoarseTs = SystemProperties.getBooleanProperty("idea.io.coarse.ts", false);
        private final LimitedPool<Memory> myMemoryPool = new LimitedPool.Sync<Memory>(10, () -> new Memory(256L));

        JnaUnixMediatorImpl() {
            assert (JnaLoader.isSupportsDirectMapping()) : "Direct mapping not available on " + Platform.RESOURCE_PREFIX;
            if ("linux-x86-64".equals(Platform.RESOURCE_PREFIX)) {
                this.myOffsets = LINUX_64;
            } else if ("darwin-x86-64".equals(Platform.RESOURCE_PREFIX)) {
                this.myOffsets = BSD_64;
            } else {
                throw new IllegalStateException("Unsupported OS/arch: " + Platform.RESOURCE_PREFIX);
            }
            Map<String, String> options = Collections.singletonMap("string-encoding", CharsetToolkit.getPlatformCharset().name());
            NativeLibrary lib = NativeLibrary.getInstance((String)"c", options);
            Native.register(LibC.class, (NativeLibrary)lib);
            Native.register(SystemInfo.isLinux ? LinuxLibC.class : UnixLibC.class, (NativeLibrary)lib);
            this.myUid = LibC.getuid();
            this.myGid = LibC.getgid();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public FileAttributes getAttributes(@NotNull String path) {
            Memory buffer = this.myMemoryPool.alloc();
            try {
                long mTime1;
                boolean isSymlink;
                int res;
                int n = res = SystemInfo.isLinux ? LinuxLibC.__lxstat64(1, path, (Pointer)buffer) : UnixLibC.lstat(path, (Pointer)buffer);
                if (res != 0) {
                    FileAttributes fileAttributes = null;
                    return fileAttributes;
                }
                int mode = this.getModeFlags(buffer) & 0xFFFF;
                boolean bl = isSymlink = (mode & 0xF000) == 40960;
                if (isSymlink) {
                    if (!JnaUnixMediatorImpl.loadFileStatus(path, buffer)) {
                        FileAttributes fileAttributes = FileAttributes.BROKEN_SYMLINK;
                        return fileAttributes;
                    }
                    mode = this.getModeFlags(buffer) & 0xFFFF;
                }
                if (DO_NOT_RESOLVE_SYMLINKS) {
                    isSymlink = false;
                }
                boolean isDirectory = (mode & 0xF000) == 16384;
                boolean isSpecial = !isDirectory && (mode & 0xF000) != 32768;
                long size = buffer.getLong((long)this.myOffsets[1]);
                long l = mTime1 = Native.LONG_SIZE == 4 ? (long)buffer.getInt((long)this.myOffsets[2]) : buffer.getLong((long)this.myOffsets[2]);
                long mTime2 = this.myCoarseTs ? 0L : (Native.LONG_SIZE == 4 ? (long)buffer.getInt((long)(this.myOffsets[2] + 4)) : buffer.getLong((long)(this.myOffsets[2] + 8)));
                long mTime = mTime1 * 1000L + mTime2 / 1000000L;
                boolean writable = this.ownFile(buffer) ? (mode & 0x92) != 0 : LibC.access(path, 2) == 0;
                FileAttributes fileAttributes = new FileAttributes(isDirectory, isSpecial, isSymlink, false, size, mTime, writable);
                return fileAttributes;
            }
            finally {
                this.myMemoryPool.recycle(buffer);
            }
        }

        @Override
        public String resolveSymLink(@NotNull String path) throws IOException {
            try {
                return DO_NOT_RESOLVE_SYMLINKS ? path : new File(path).getCanonicalPath();
            }
            catch (IOException e) {
                String message = e.getMessage();
                if (message != null && StringUtil.toLowerCase(message).contains("too many levels of symbolic links")) {
                    LOG.debug(e);
                    return null;
                }
                throw new IOException("Cannot resolve '" + path + "'", e);
            }
        }

        private static boolean loadFileStatus(@NotNull String path, @NotNull Memory buffer) {
            return (SystemInfo.isLinux ? LinuxLibC.__xstat64(1, path, (Pointer)buffer) : UnixLibC.stat(path, (Pointer)buffer)) == 0;
        }

        private int getModeFlags(@NotNull Memory buffer) {
            return SystemInfo.isLinux ? buffer.getInt((long)this.myOffsets[0]) : (int)buffer.getShort((long)this.myOffsets[0]);
        }

        private boolean ownFile(@NotNull Memory buffer) {
            return buffer.getInt((long)this.myOffsets[3]) == this.myUid && buffer.getInt((long)this.myOffsets[4]) == this.myGid;
        }

        private static final class LibC {
            static final int S_MASK = 65535;
            static final int S_IFMT = 61440;
            static final int S_IFLNK = 40960;
            static final int S_IFREG = 32768;
            static final int S_IFDIR = 16384;
            static final int WRITE_MASK = 146;
            static final int W_OK = 2;

            private LibC() {
            }

            static native int getuid();

            static native int getgid();

            static native int access(String var0, int var1);
        }

        private static final class LinuxLibC {
            private LinuxLibC() {
            }

            static native int __lxstat64(int var0, String var1, Pointer var2);

            static native int __xstat64(int var0, String var1, Pointer var2);
        }

        private static final class UnixLibC {
            private UnixLibC() {
            }

            static native int lstat(String var0, Pointer var1);

            static native int stat(String var0, Pointer var1);
        }
    }

    static interface Mediator {
        @Nullable
        public FileAttributes getAttributes(@NotNull String var1) throws IOException;

        @Nullable
        public String resolveSymLink(@NotNull String var1) throws IOException;
    }

    private static class Nio2MediatorImpl
    implements Mediator {
        private Nio2MediatorImpl() {
        }

        @Override
        public FileAttributes getAttributes(@NotNull String pathStr) {
            try {
                Path path = Paths.get(pathStr, new String[0]);
                BasicFileAttributes attributes = NioFiles.readAttributes(path);
                return attributes == NioFiles.BROKEN_SYMLINK ? FileAttributes.BROKEN_SYMLINK : FileAttributes.fromNio(path, attributes);
            }
            catch (IOException | InvalidPathException e) {
                LOG.debug(pathStr, e);
                return null;
            }
        }

        @Override
        public String resolveSymLink(@NotNull String path) throws IOException {
            try {
                return Paths.get(path, new String[0]).toRealPath(new LinkOption[0]).toString();
            }
            catch (FileSystemException e) {
                LOG.debug(path, e);
                return null;
            }
        }
    }

    private static interface NtOsKrnl
    extends StdCallLibrary,
    WinNT {
        public static final NtOsKrnl INSTANCE = (NtOsKrnl)Native.load((String)"NtDll", NtOsKrnl.class, (Map)W32APIOptions.UNICODE_OPTIONS);
        public static final int FILE_SHARE_ALL = 7;
        public static final int FileCaseSensitiveInformation = 71;

        public int NtQueryInformationFile(WinNT.HANDLE var1, IO_STATUS_BLOCK_P var2, Structure var3, long var4, int var6);

        @Structure.FieldOrder(value={"Flags"})
        public static class FILE_CASE_SENSITIVE_INFORMATION_P
        extends Structure
        implements Structure.ByReference {
            public long Flags = 0xFFFFFFFFL;
        }

        @Structure.FieldOrder(value={"Pointer", "Information"})
        public static class IO_STATUS_BLOCK_P
        extends Structure
        implements Structure.ByReference {
            public Pointer Pointer;
            public Pointer Information;
        }
    }

    private static interface CoreFoundation
    extends com.sun.jna.platform.mac.CoreFoundation {
        public static final CoreFoundation INSTANCE = (CoreFoundation)Native.load((String)"CoreFoundation", CoreFoundation.class);
        public static final CoreFoundation.CFStringRef kCFURLVolumeSupportsCaseSensitiveNamesKey = CoreFoundation.CFStringRef.createCFString((String)"NSURLVolumeSupportsCaseSensitiveNamesKey");

        public CoreFoundation.CFTypeRef CFURLCreateFromFileSystemRepresentation(CoreFoundation.CFAllocatorRef var1, byte[] var2, long var3, boolean var5);

        public boolean CFURLCopyResourcePropertyForKey(CoreFoundation.CFTypeRef var1, CoreFoundation.CFStringRef var2, PointerByReference var3, PointerByReference var4);

        public CoreFoundation.CFStringRef CFErrorGetDomain(Pointer var1);

        public CoreFoundation.CFIndex CFErrorGetCode(Pointer var1);
    }

    private static interface LibC
    extends Library {
        public static final LibC INSTANCE = (LibC)Native.load(LibC.class);

        public int statfs(String var1, Memory var2);
    }

    static interface E2P
    extends Library {
        public static final E2P INSTANCE = (E2P)Native.load((String)"e2p", E2P.class);
        public static final long EXT4_CASEFOLD_FL = 0x40000000L;

        public int fgetflags(String var1, LongByReference var2);
    }
}

