/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.adservices.ondevicepersonalization;

import android.annotation.NonNull;
import android.annotation.Nullable;
import android.content.ComponentName;
import android.net.Uri;
import android.os.Parcelable;

import com.android.ondevicepersonalization.internal.util.AnnotationValidations;
import com.android.ondevicepersonalization.internal.util.DataClass;

/**
 * A class that contains Web Trigger Event data sent from the Measurement API to the
 * OnDevicePersonalization service when the browser registers a web trigger
 * with the <a href="https://developer.android.com/design-for-safety/privacy-sandbox/guides/attribution">
 * Measurement API</a> and the web trigger data is intended to be processed by an
 * {@link IsolatedService}.
 * @hide
 */
@DataClass(genAidl = false, genBuilder = false)
public final class MeasurementWebTriggerEventParamsParcel implements Parcelable {
    /**
     * The URL of the web page where the web trigger event occurred.
     */
    @NonNull private Uri mDestinationUrl;

    /**
     * The package name of the browser app where the web trigger event occurred.
     */
    @NonNull private String mAppPackageName;

    /**
     * The package and class name of the {@link IsolatedService} that should process
     * the web trigger event.
     */
    @NonNull private ComponentName mIsolatedService;

    /**
     * An optional SHA-256 hash of the signing key of the package that contains
     * the {@link IsolatedService}, to guard against package name spoofing via sideloading.
     * If this field is present and does not match the signing key of the installed receiver
     * service package, the web trigger event is discarded.
     */
    @Nullable private String mCertDigest = null;

    /**
     * Additional data that the server may provide to the {@link IsolatedService}. This can be
     * {@code null} if the server does not need to provide any data other than the required fields.
     */
    @Nullable private byte[] mEventData = null;

    public MeasurementWebTriggerEventParamsParcel(
            @NonNull MeasurementWebTriggerEventParams params) {
        this(params.getDestinationUrl(), params.getAppPackageName(), params.getIsolatedService(),
                params.getCertDigest(), params.getEventData());
    }



    // Code below generated by codegen v1.0.23.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen $ANDROID_BUILD_TOP/packages/modules/OnDevicePersonalization/framework/java/android/adservices/ondevicepersonalization/MeasurementWebTriggerEventParamsParcel.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    //@formatter:off


    /**
     * Creates a new MeasurementWebTriggerEventParamsParcel.
     *
     * @param destinationUrl
     *   The URL of the web page where the web trigger event occurred.
     * @param appPackageName
     *   The package name of the browser app where the web trigger event occurred.
     * @param isolatedService
     *   The package and class name of the {@link IsolatedService} that should process
     *   the web trigger event.
     * @param certDigest
     *   An optional SHA-256 hash of the signing key of the package that contains
     *   the {@link IsolatedService}, to guard against package name spoofing via sideloading.
     *   If this field is present and does not match the signing key of the installed receiver
     *   service package, the web trigger event is discarded.
     * @param eventData
     *   Additional data that the server may provide to the {@link IsolatedService}. This can be
     *   {@code null} if the server does not need to provide any data other than the required fields.
     */
    @DataClass.Generated.Member
    public MeasurementWebTriggerEventParamsParcel(
            @NonNull Uri destinationUrl,
            @NonNull String appPackageName,
            @NonNull ComponentName isolatedService,
            @Nullable String certDigest,
            @Nullable byte[] eventData) {
        this.mDestinationUrl = destinationUrl;
        AnnotationValidations.validate(
                NonNull.class, null, mDestinationUrl);
        this.mAppPackageName = appPackageName;
        AnnotationValidations.validate(
                NonNull.class, null, mAppPackageName);
        this.mIsolatedService = isolatedService;
        AnnotationValidations.validate(
                NonNull.class, null, mIsolatedService);
        this.mCertDigest = certDigest;
        this.mEventData = eventData;

        // onConstructed(); // You can define this method to get a callback
    }

    /**
     * The URL of the web page where the web trigger event occurred.
     */
    @DataClass.Generated.Member
    public @NonNull Uri getDestinationUrl() {
        return mDestinationUrl;
    }

    /**
     * The package name of the browser app where the web trigger event occurred.
     */
    @DataClass.Generated.Member
    public @NonNull String getAppPackageName() {
        return mAppPackageName;
    }

    /**
     * The package and class name of the {@link IsolatedService} that should process
     * the web trigger event.
     */
    @DataClass.Generated.Member
    public @NonNull ComponentName getIsolatedService() {
        return mIsolatedService;
    }

    /**
     * An optional SHA-256 hash of the signing key of the package that contains
     * the {@link IsolatedService}, to guard against package name spoofing via sideloading.
     * If this field is present and does not match the signing key of the installed receiver
     * service package, the web trigger event is discarded.
     */
    @DataClass.Generated.Member
    public @Nullable String getCertDigest() {
        return mCertDigest;
    }

    /**
     * Additional data that the server may provide to the {@link IsolatedService}. This can be
     * {@code null} if the server does not need to provide any data other than the required fields.
     */
    @DataClass.Generated.Member
    public @Nullable byte[] getEventData() {
        return mEventData;
    }

    @Override
    @DataClass.Generated.Member
    public void writeToParcel(@NonNull android.os.Parcel dest, int flags) {
        // You can override field parcelling by defining methods like:
        // void parcelFieldName(Parcel dest, int flags) { ... }

        byte flg = 0;
        if (mCertDigest != null) flg |= 0x8;
        dest.writeByte(flg);
        dest.writeTypedObject(mDestinationUrl, flags);
        dest.writeString(mAppPackageName);
        dest.writeTypedObject(mIsolatedService, flags);
        if (mCertDigest != null) dest.writeString(mCertDigest);
        dest.writeByteArray(mEventData);
    }

    @Override
    @DataClass.Generated.Member
    public int describeContents() { return 0; }

    /** @hide */
    @SuppressWarnings({"unchecked", "RedundantCast"})
    @DataClass.Generated.Member
    /* package-private */ MeasurementWebTriggerEventParamsParcel(@NonNull android.os.Parcel in) {
        // You can override field unparcelling by defining methods like:
        // static FieldType unparcelFieldName(Parcel in) { ... }

        byte flg = in.readByte();
        Uri destinationUrl = (Uri) in.readTypedObject(Uri.CREATOR);
        String appPackageName = in.readString();
        ComponentName isolatedService = (ComponentName) in.readTypedObject(ComponentName.CREATOR);
        String certDigest = (flg & 0x8) == 0 ? null : in.readString();
        byte[] eventData = in.createByteArray();

        this.mDestinationUrl = destinationUrl;
        AnnotationValidations.validate(
                NonNull.class, null, mDestinationUrl);
        this.mAppPackageName = appPackageName;
        AnnotationValidations.validate(
                NonNull.class, null, mAppPackageName);
        this.mIsolatedService = isolatedService;
        AnnotationValidations.validate(
                NonNull.class, null, mIsolatedService);
        this.mCertDigest = certDigest;
        this.mEventData = eventData;

        // onConstructed(); // You can define this method to get a callback
    }

    @DataClass.Generated.Member
    public static final @NonNull Parcelable.Creator<MeasurementWebTriggerEventParamsParcel> CREATOR
            = new Parcelable.Creator<MeasurementWebTriggerEventParamsParcel>() {
        @Override
        public MeasurementWebTriggerEventParamsParcel[] newArray(int size) {
            return new MeasurementWebTriggerEventParamsParcel[size];
        }

        @Override
        public MeasurementWebTriggerEventParamsParcel createFromParcel(@NonNull android.os.Parcel in) {
            return new MeasurementWebTriggerEventParamsParcel(in);
        }
    };

    @DataClass.Generated(
            time = 1707510209072L,
            codegenVersion = "1.0.23",
            sourceFile = "packages/modules/OnDevicePersonalization/framework/java/android/adservices/ondevicepersonalization/MeasurementWebTriggerEventParamsParcel.java",
            inputSignatures = "private @android.annotation.NonNull android.net.Uri mDestinationUrl\nprivate @android.annotation.NonNull java.lang.String mAppPackageName\nprivate @android.annotation.NonNull android.content.ComponentName mIsolatedService\nprivate @android.annotation.Nullable java.lang.String mCertDigest\nprivate @android.annotation.Nullable byte[] mEventData\nclass MeasurementWebTriggerEventParamsParcel extends java.lang.Object implements [android.os.Parcelable]\n@com.android.ondevicepersonalization.internal.util.DataClass(genAidl=false, genBuilder=false)")
    @Deprecated
    private void __metadata() {}


    //@formatter:on
    // End of generated code

}
