/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.adservices.ondevicepersonalization;

import android.annotation.NonNull;
import android.annotation.Nullable;
import android.os.Parcelable;

import com.android.ondevicepersonalization.internal.util.AnnotationValidations;
import com.android.ondevicepersonalization.internal.util.DataClass;

import java.util.Collections;
import java.util.List;

/**
 * Parcelable version of {@link ExecuteOutput}.
 * @hide
 */
@DataClass(genAidl = false, genBuilder = false)
public final class ExecuteOutputParcel implements Parcelable {
    /**
     * Persistent data to be written to the REQUESTS table after
     * {@link IsolatedWorker#onExecute(ExecuteInput, android.os.OutcomeReceiver)}
     * completes. If null, no persistent data will be written.
     */
    @Nullable private RequestLogRecord mRequestLogRecord = null;

    /**
     * A {@link RenderingConfig} object that contains information about the content to be rendered
     * in the client app view. Can be null if no content is to be rendered.
     */
    @Nullable private RenderingConfig mRenderingConfig = null;

    /**
     * A list of {@link EventLogRecord}. Writes events to the EVENTS table and associates
     * them with requests with the specified corresponding {@link RequestLogRecord} from
     * {@link EventLogRecord#getRequestLogRecord()}.
     * If the event does not contain a {@link RequestLogRecord} emitted by this package, the
     * EventLogRecord is not written.
     *
     */
    @DataClass.PluralOf("eventLogRecord")
    @NonNull private List<EventLogRecord> mEventLogRecords = Collections.emptyList();

    /**
     * A byte array returned by an {@link IsolatedService} to a calling app. The contents of
     * this array is returned to the caller of
     * {@link OnDevicePersonalizationManager#execute(ComponentName, PersistableBundle, java.util.concurrent.Executor, OutcomeReceiver)}
     * if the (calling app package, isolated service package) pair is present in an allow list
     * that permits data to be returned to the caller.
     *
     * @hide
     */
    @Nullable private byte[] mOutputData = null;

    /**
     * An integer value that an {@link IsolatedService} may optionally return to to a calling app,
     * by setting this field to the value between 0 and max value in {@link
     * ExecuteInIsolatedServiceRequest.Options}. The value of this field will be returned to the
     * caller of {@link OnDevicePersonalizationManager#executeInIsolatedService} if returning data
     * from isolated processes is allowed by policy and the (calling app package, isolated service
     * package) pair is present in an allowlist that permits data to be returned.
     *
     * @hide
     */
    private int mBestValue = -1;

    /** @hide */
    public ExecuteOutputParcel(@NonNull ExecuteOutput value) {
        this(
                value.getRequestLogRecord(),
                value.getRenderingConfig(),
                value.getEventLogRecords(),
                value.getOutputData(),
                value.getBestValue());
    }

    // Code below generated by codegen v1.0.23.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen
    // $ANDROID_BUILD_TOP/packages/modules/OnDevicePersonalization/framework/java/android/adservices/ondevicepersonalization/ExecuteOutputParcel.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    // @formatter:off

    /**
     * Creates a new ExecuteOutputParcel.
     *
     * @param requestLogRecord Persistent data to be written to the REQUESTS table after {@link
     *     IsolatedWorker#onExecute(ExecuteInput, android.os.OutcomeReceiver)} completes. If null,
     *     no persistent data will be written.
     * @param renderingConfig A {@link RenderingConfig} object that contains information about the
     *     content to be rendered in the client app view. Can be null if no content is to be
     *     rendered.
     * @param eventLogRecords A list of {@link EventLogRecord}. Writes events to the EVENTS table
     *     and associates them with requests with the specified corresponding {@link
     *     RequestLogRecord} from {@link EventLogRecord#getRequestLogRecord()}. If the event does
     *     not contain a {@link RequestLogRecord} emitted by this package, the EventLogRecord is not
     *     written.
     * @param outputData A byte array returned by an {@link IsolatedService} to a calling app. The
     *     contents of this array is returned to the caller of {@link
     *     OnDevicePersonalizationManager#execute(ComponentName, PersistableBundle,
     *     java.util.concurrent.Executor, OutcomeReceiver)} if the (calling app package, isolated
     *     service package) pair is present in an allow list that permits data to be returned to the
     *     caller.
     * @param bestValue An integer value that an {@link IsolatedService} may optionally return to to
     *     a calling app, by setting this field to the value between 0 and max value in {@link
     *     ExecuteInIsolatedServiceRequest.OutputSpec}. The value of this field will be returned to
     *     the caller of {@link OnDevicePersonalizationManager#executeInIsolatedService} if
     *     returning data from isolated processes is allowed by policy and the (calling app package,
     *     isolated service package) pair is present in an allowlist that permits data to be
     *     returned.
     */
    @DataClass.Generated.Member
    public ExecuteOutputParcel(
            @Nullable RequestLogRecord requestLogRecord,
            @Nullable RenderingConfig renderingConfig,
            @NonNull List<EventLogRecord> eventLogRecords,
            @Nullable byte[] outputData,
            int bestValue) {
        this.mRequestLogRecord = requestLogRecord;
        this.mRenderingConfig = renderingConfig;
        this.mEventLogRecords = eventLogRecords;
        AnnotationValidations.validate(
                NonNull.class, null, mEventLogRecords);
        this.mOutputData = outputData;
        this.mBestValue = bestValue;

        // onConstructed(); // You can define this method to get a callback
    }

    /**
     * Persistent data to be written to the REQUESTS table after
     * {@link IsolatedWorker#onExecute(ExecuteInput, android.os.OutcomeReceiver)}
     * completes. If null, no persistent data will be written.
     */
    @DataClass.Generated.Member
    public @Nullable RequestLogRecord getRequestLogRecord() {
        return mRequestLogRecord;
    }

    /**
     * A {@link RenderingConfig} object that contains information about the content to be rendered
     * in the client app view. Can be null if no content is to be rendered.
     */
    @DataClass.Generated.Member
    public @Nullable RenderingConfig getRenderingConfig() {
        return mRenderingConfig;
    }

    /**
     * A list of {@link EventLogRecord}. Writes events to the EVENTS table and associates
     * them with requests with the specified corresponding {@link RequestLogRecord} from
     * {@link EventLogRecord#getRequestLogRecord()}.
     * If the event does not contain a {@link RequestLogRecord} emitted by this package, the
     * EventLogRecord is not written.
     */
    @DataClass.Generated.Member
    public @NonNull List<EventLogRecord> getEventLogRecords() {
        return mEventLogRecords;
    }

    /**
     * A byte array returned by an {@link IsolatedService} to a calling app. The contents of
     * this array is returned to the caller of
     * {@link OnDevicePersonalizationManager#execute(ComponentName, PersistableBundle, java.util.concurrent.Executor, OutcomeReceiver)}
     * if the (calling app package, isolated service package) pair is present in an allow list
     * that permits data to be returned to the caller.
     *
     * @hide
     */
    @DataClass.Generated.Member
    public @Nullable byte[] getOutputData() {
        return mOutputData;
    }

    /**
     * An integer value that an {@link IsolatedService} may optionally return to to a calling app,
     * by setting this field to the value between 0 and max value in {@link
     * ExecuteInIsolatedServiceRequest.Options}. The value of this field will be returned to the
     * caller of {@link OnDevicePersonalizationManager#executeInIsolatedService} if returning data
     * from isolated processes is allowed by policy and the (calling app package, isolated service
     * package) pair is present in an allowlist that permits data to be returned.
     *
     * @hide
     */
    @DataClass.Generated.Member
    public int getBestValue() {
        return mBestValue;
    }

    @Override
    @DataClass.Generated.Member
    public void writeToParcel(@NonNull android.os.Parcel dest, int flags) {
        // You can override field parcelling by defining methods like:
        // void parcelFieldName(Parcel dest, int flags) { ... }

        byte flg = 0;
        if (mRequestLogRecord != null) flg |= 0x1;
        if (mRenderingConfig != null) flg |= 0x2;
        dest.writeByte(flg);
        if (mRequestLogRecord != null) dest.writeTypedObject(mRequestLogRecord, flags);
        if (mRenderingConfig != null) dest.writeTypedObject(mRenderingConfig, flags);
        dest.writeParcelableList(mEventLogRecords, flags);
        dest.writeByteArray(mOutputData);
        dest.writeInt(mBestValue);
    }

    @Override
    @DataClass.Generated.Member
    public int describeContents() { return 0; }

    /** @hide */
    @SuppressWarnings({"unchecked", "RedundantCast"})
    @DataClass.Generated.Member
    /* package-private */ ExecuteOutputParcel(@NonNull android.os.Parcel in) {
        // You can override field unparcelling by defining methods like:
        // static FieldType unparcelFieldName(Parcel in) { ... }

        byte flg = in.readByte();
        RequestLogRecord requestLogRecord = (flg & 0x1) == 0 ? null : (RequestLogRecord) in.readTypedObject(RequestLogRecord.CREATOR);
        RenderingConfig renderingConfig = (flg & 0x2) == 0 ? null : (RenderingConfig) in.readTypedObject(RenderingConfig.CREATOR);
        List<EventLogRecord> eventLogRecords = new java.util.ArrayList<>();
        in.readParcelableList(eventLogRecords, EventLogRecord.class.getClassLoader());
        byte[] outputData = in.createByteArray();
        int bestValue = in.readInt();

        this.mRequestLogRecord = requestLogRecord;
        this.mRenderingConfig = renderingConfig;
        this.mEventLogRecords = eventLogRecords;
        AnnotationValidations.validate(
                NonNull.class, null, mEventLogRecords);
        this.mOutputData = outputData;
        this.mBestValue = bestValue;

        // onConstructed(); // You can define this method to get a callback
    }

    @DataClass.Generated.Member
    public static final @NonNull Parcelable.Creator<ExecuteOutputParcel> CREATOR
            = new Parcelable.Creator<ExecuteOutputParcel>() {
        @Override
        public ExecuteOutputParcel[] newArray(int size) {
            return new ExecuteOutputParcel[size];
        }

        @Override
        public ExecuteOutputParcel createFromParcel(@NonNull android.os.Parcel in) {
            return new ExecuteOutputParcel(in);
        }
    };

    @DataClass.Generated(
            time = 1721773162236L,
            codegenVersion = "1.0.23",
            sourceFile =
                    "packages/modules/OnDevicePersonalization/framework/java/android/adservices/ondevicepersonalization/ExecuteOutputParcel.java",
            inputSignatures =
                    "private @android.annotation.Nullable android.adservices.ondevicepersonalization.RequestLogRecord mRequestLogRecord\nprivate @android.annotation.Nullable android.adservices.ondevicepersonalization.RenderingConfig mRenderingConfig\nprivate @com.android.ondevicepersonalization.internal.util.DataClass.PluralOf(\"eventLogRecord\") @android.annotation.NonNull java.util.List<android.adservices.ondevicepersonalization.EventLogRecord> mEventLogRecords\nprivate @android.annotation.Nullable byte[] mOutputData\nprivate  int mBestValue\nclass ExecuteOutputParcel extends java.lang.Object implements [android.os.Parcelable]\n@com.android.ondevicepersonalization.internal.util.DataClass(genAidl=false, genBuilder=false)")
    @Deprecated
    private void __metadata() {}

    //@formatter:on
    // End of generated code

}
