/*
 * Copyright (C) 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.adservices.common;

import static com.android.adservices.flags.Flags.FLAG_FLEDGE_ENABLE_CUSTOM_AUDIENCE_COMPONENT_ADS;

import android.annotation.FlaggedApi;
import android.annotation.NonNull;
import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;

import java.util.Objects;

/**
 * Represents data specific to a component ad that is necessary for ad selection and rendering. This
 * is to support use case for ads composed of multiple pieces, such as an ad displaying multiple
 * products at once.
 */
@FlaggedApi(FLAG_FLEDGE_ENABLE_CUSTOM_AUDIENCE_COMPONENT_ADS)
public final class ComponentAdData implements Parcelable {
    private final Uri mRenderUri;
    private final String mAdRenderId;

    @NonNull
    public static final Creator<ComponentAdData> CREATOR =
            new Creator<>() {
                @Override
                public ComponentAdData createFromParcel(@NonNull Parcel in) {
                    return new ComponentAdData(in);
                }

                @Override
                public ComponentAdData[] newArray(int size) {
                    return new ComponentAdData[size];
                }
            };

    private ComponentAdData(@NonNull Parcel in) {
        mRenderUri = Uri.CREATOR.createFromParcel(in);
        mAdRenderId = in.readString();
    }

    /**
     * Constructs a {@link ComponentAdData} object.
     *
     * <p>This object represents a single component within a multi-part ad. The provided render URI
     * points to the resources needed to render this component. Both the URI and the resources to
     * render the component are provided by the buyer ad tech.
     *
     * @param renderUri the URI that points to the component's rendering resources.
     * @param adRenderId the component ad render ID used in the auction flow to identify the ad and
     *     bid on it. This ID is typically generated by the buyer ad tech and is also used to track
     *     the individual component within the larger ad.
     * @throws IllegalArgumentException if the provided {@code adRenderId} is empty.
     */
    public ComponentAdData(@NonNull Uri renderUri, @NonNull String adRenderId) {
        if (adRenderId.isEmpty()) {
            throw new IllegalArgumentException("Ad render id cannot be empty");
        }
        mRenderUri = Objects.requireNonNull(renderUri, "Provided render uri is null");
        mAdRenderId = Objects.requireNonNull(adRenderId, "Provided ad render id is null");
    }

    @Override
    public void writeToParcel(@NonNull Parcel dest, int flags) {
        Objects.requireNonNull(dest);

        mRenderUri.writeToParcel(dest, flags);
        dest.writeString(mAdRenderId);
    }

    @Override
    public int describeContents() {
        return 0;
    }

    /**
     * Gets the URI that points to the component's rendering resources.
     *
     * <p>Both the URI and the resources it points to are provided by the buyer ad tech.
     */
    @NonNull
    public Uri getRenderUri() {
        return mRenderUri;
    }

    /**
     * Gets the component ad render id for server auctions.
     *
     * <p>Ad render id is collected for each {@link ComponentAdData} when server auction request is
     * received.
     *
     * <p>The overall size of the Custom Audience is limited. The size of this field is considered
     * using {@link String#getBytes()} in {@code UTF-8} encoding.
     */
    @NonNull
    public String getAdRenderId() {
        return mAdRenderId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof ComponentAdData adData)) return false;
        return mRenderUri.equals(adData.mRenderUri)
                && Objects.equals(mAdRenderId, adData.mAdRenderId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(mRenderUri, mAdRenderId);
    }

    @Override
    public String toString() {
        return "ComponentAdData{"
                + "mRenderUri="
                + mRenderUri
                + ", mAdRenderId='"
                + mAdRenderId
                + '\''
                + '}';
    }
}
