/*
 * Copyright (C) 2018 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.net.wifi;

import android.net.ConnectivityManager;
import android.net.ConnectivityManager.NetworkCallback;
import android.net.MacAddress;
import android.net.NetworkCapabilities;
import android.net.NetworkRequest;
import android.net.NetworkSpecifier;
import android.os.PatternMatcher;

/**
 * Network specifier object used to request a Wi-Fi network. Apps should use the
 * {@link android.net.wifi.WifiNetworkSpecifier.Builder WifiNetworkSpecifier.Builder} class to create an instance.
 * <p>
 * This specifier can be used to request a local-only connection on devices that support concurrent
 * connections (indicated via
 * {@link android.net.wifi.WifiManager#isStaConcurrencyForLocalOnlyConnectionsSupported() WifiManager.isStaConcurrencyForLocalOnlyConnectionsSupported()} and if the initiating app
 * targets SDK &ge; {@link android.os.Build.VERSION_CODES#S} or is a system app. These local-only
 * connections may be brought up as a secondary concurrent connection (primary connection will be
 * used for networks with internet connectivity available to the user and all apps).
 * </p>
 * <p>
 * This specifier can also be used to listen for connected Wi-Fi networks on a particular band.
 * Additionally, some devices may support requesting a connection to a particular band. If the
 * device does not support such a request, it will send {@link android.net.ConnectivityManager.NetworkCallback#onUnavailable() NetworkCallback.onUnavailable()}
 * upon request to the callback passed to
 * {@link android.net.ConnectivityManager#requestNetwork(android.net.NetworkRequest,android.net.ConnectivityManager.NetworkCallback) ConnectivityManager.requestNetwork(NetworkRequest, NetworkCallback)} or equivalent.
 * See {@link android.net.wifi.WifiNetworkSpecifier.Builder#build() Builder.build()} for details.
 * </p>
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class WifiNetworkSpecifier extends android.net.NetworkSpecifier implements android.os.Parcelable {

WifiNetworkSpecifier() { throw new RuntimeException("Stub!"); }

/** @hide */

public boolean canBeSatisfiedBy(android.net.NetworkSpecifier other) { throw new RuntimeException("Stub!"); }

public int describeContents() { throw new RuntimeException("Stub!"); }

public boolean equals(java.lang.Object obj) { throw new RuntimeException("Stub!"); }

/**
 * The band for this Wi-Fi network specifier.

 * @return Value is {@link android.net.wifi.ScanResult#UNSPECIFIED}, {@link android.net.wifi.ScanResult#WIFI_BAND_24_GHZ}, {@link android.net.wifi.ScanResult#WIFI_BAND_5_GHZ}, {@link android.net.wifi.ScanResult#WIFI_BAND_6_GHZ}, or {@link android.net.wifi.ScanResult#WIFI_BAND_60_GHZ}
 */

public int getBand() { throw new RuntimeException("Stub!"); }

/**
 * The preferred channels fot this network specifier.
 * @see android.net.wifi.WifiNetworkSpecifier.Builder#setPreferredChannelsFrequenciesMhz(int[])

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public int[] getPreferredChannelFrequenciesMhz() { throw new RuntimeException("Stub!"); }

public int hashCode() { throw new RuntimeException("Stub!"); }

/** @hide */

@android.annotation.Nullable
public android.net.NetworkSpecifier redact() { throw new RuntimeException("Stub!"); }

public java.lang.String toString() { throw new RuntimeException("Stub!"); }

public void writeToParcel(android.os.Parcel dest, int flags) { throw new RuntimeException("Stub!"); }

@android.annotation.NonNull public static final android.os.Parcelable.Creator<android.net.wifi.WifiNetworkSpecifier> CREATOR;
static { CREATOR = null; }
/**
 * Builder used to create {@link android.net.wifi.WifiNetworkSpecifier WifiNetworkSpecifier} objects.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class Builder {

public Builder() { throw new RuntimeException("Stub!"); }

/**
 * Create a specifier object used to request a Wi-Fi network. The generated
 * {@link android.net.NetworkSpecifier NetworkSpecifier} should be used in
 * {@link android.net.NetworkRequest.Builder#setNetworkSpecifier(android.net.NetworkSpecifier) NetworkRequest.Builder.setNetworkSpecifier(NetworkSpecifier)} when building
 * the {@link android.net.NetworkRequest NetworkRequest}.
 *
 *<p>
 * When using with {@link android.net.ConnectivityManager#requestNetwork(android.net.NetworkRequest,android.net.ConnectivityManager.NetworkCallback) ConnectivityManager.requestNetwork(NetworkRequest,
 * NetworkCallback)} or variants, note that some devices may not support requesting a
 * network with all combinations of specifier members. For example, some devices may only
 * support requesting local-only networks (networks without the
 * {@link android.net.NetworkCapabilities#NET_CAPABILITY_INTERNET NetworkCapabilities.NET_CAPABILITY_INTERNET} capability), or not support
 * requesting a particular band. However, there are no restrictions when using
 * {@link android.net.ConnectivityManager#registerNetworkCallback(android.net.NetworkRequest,android.net.ConnectivityManager.NetworkCallback) ConnectivityManager.registerNetworkCallback(NetworkRequest, NetworkCallback)}
 * or other similar methods which monitor but do not request networks.
 *
 * If the device can't support a request, the app will receive a call to
 * {@link android.net.ConnectivityManager.NetworkCallback#onUnavailable() NetworkCallback.onUnavailable()}.
 *</p>
 *
 *<p>
 * When requesting a local-only network, apps can set a combination of network match params:
 * <li> SSID Pattern using {@link #setSsidPattern(android.os.PatternMatcher)} OR Specific SSID using
 * {@link #setSsid(java.lang.String)}. </li>
 * AND/OR
 * <li> BSSID Pattern using {@link #setBssidPattern(android.net.MacAddress,android.net.MacAddress)} OR Specific
 * BSSID using {@link #setBssid(android.net.MacAddress)} </li>
 * to trigger connection to a network that matches the set params.
 * The system will find the set of networks matching the request and present the user
 * with a system dialog which will allow the user to select a specific Wi-Fi network to
 * connect to or to deny the request.
 *
 * To protect user privacy, some limitations to the ability of matching patterns apply.
 * In particular, when the system brings up a network to satisfy a {@link android.net.NetworkRequest NetworkRequest}
 * from some app, the system reserves the right to decline matching the SSID pattern to
 * the real SSID of the network for other apps than the app that requested the network, and
 * not send those callbacks even if the SSID matches the requested pattern.
 *</p>
 *
 * For example:
 * To connect to an open network with a SSID prefix of "test" and a BSSID OUI of "10:03:23":
 *
 * <pre>{@code
 * final NetworkSpecifier specifier =
 *      new Builder()
 *      .setSsidPattern(new PatternMatcher("test", PatternMatcher.PATTERN_PREFIX))
 *      .setBssidPattern(MacAddress.fromString("10:03:23:00:00:00"),
 *                       MacAddress.fromString("ff:ff:ff:00:00:00"))
 *      .build()
 * final NetworkRequest request =
 *      new NetworkRequest.Builder()
 *      .addTransportType(NetworkCapabilities.TRANSPORT_WIFI)
 *      .removeCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET)
 *      .setNetworkSpecifier(specifier)
 *      .build();
 * final ConnectivityManager connectivityManager =
 *      context.getSystemService(Context.CONNECTIVITY_SERVICE);
 * final NetworkCallback networkCallback = new NetworkCallback() {
 *      ...
 *      {@literal @}Override
 *      void onAvailable(...) {}
 *      // etc.
 * };
 * connectivityManager.requestNetwork(request, networkCallback);
 * }</pre>
 *
 * @return Instance of {@link android.net.NetworkSpecifier NetworkSpecifier}.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalStateException on invalid params set.
 */

@android.annotation.NonNull
public android.net.wifi.WifiNetworkSpecifier build() { throw new RuntimeException("Stub!"); }

/**
 * Specifies the band requested for this network.
 *
 * Only a single band can be requested. An app can file multiple callbacks concurrently
 * if they need to know about multiple bands.
 *
 * @param band The requested band.
 * Value is {@link android.net.wifi.ScanResult#UNSPECIFIED}, {@link android.net.wifi.ScanResult#WIFI_BAND_24_GHZ}, {@link android.net.wifi.ScanResult#WIFI_BAND_5_GHZ}, {@link android.net.wifi.ScanResult#WIFI_BAND_6_GHZ}, or {@link android.net.wifi.ScanResult#WIFI_BAND_60_GHZ}
 * @return Instance of {@link android.net.wifi.WifiNetworkSpecifier.Builder Builder} to enable chaining of the builder method.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.WifiNetworkSpecifier.Builder setBand(int band) { throw new RuntimeException("Stub!"); }

/**
 * Set the BSSID to use for filtering networks from scan results. Will only match network
 * whose BSSID is identical to the specified value.
 * <p>
 * <li>Sets the BSSID to use for filtering networks from scan results. Will only match
 * networks whose BSSID is identical to specified value.</li>
 * <li>Overrides any previous value set using {@link #setBssid(android.net.MacAddress)} or
 * {@link #setBssidPattern(android.net.MacAddress,android.net.MacAddress)}.</li>
 *
 * @param bssid BSSID of the network.
 * This value cannot be {@code null}.
 * @return Instance of {@link android.net.wifi.WifiNetworkSpecifier.Builder Builder} to enable chaining of the builder method.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.WifiNetworkSpecifier.Builder setBssid(@android.annotation.NonNull android.net.MacAddress bssid) { throw new RuntimeException("Stub!"); }

/**
 * Set the BSSID match pattern to use for filtering networks from scan results.
 * Will match all networks with BSSID which satisfies the following:
 * {@code BSSID & mask == baseAddress}.
 * <p>
 * <li>Overrides any previous value set using {@link #setBssid(android.net.MacAddress)} or
 * {@link #setBssidPattern(android.net.MacAddress,android.net.MacAddress)}.</li>
 *
 * @param baseAddress Base address for BSSID pattern.
 * This value cannot be {@code null}.
 * @param mask Mask for BSSID pattern.
 * This value cannot be {@code null}.
 * @return Instance of {@link android.net.wifi.WifiNetworkSpecifier.Builder Builder} to enable chaining of the builder method.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.WifiNetworkSpecifier.Builder setBssidPattern(@android.annotation.NonNull android.net.MacAddress baseAddress, @android.annotation.NonNull android.net.MacAddress mask) { throw new RuntimeException("Stub!"); }

/**
 * Specifies whether this represents an Enhanced Open (OWE) network.
 *
 * @param isEnhancedOpen {@code true} to indicate that the network uses enhanced open,
 *                       {@code false} otherwise.
 * @return Instance of {@link android.net.wifi.WifiNetworkSpecifier.Builder Builder} to enable chaining of the builder method.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.WifiNetworkSpecifier.Builder setIsEnhancedOpen(boolean isEnhancedOpen) { throw new RuntimeException("Stub!"); }

/**
 * Specifies whether this represents a hidden network.
 * <p>
 * <li>Setting this disallows the usage of {@link #setSsidPattern(android.os.PatternMatcher)} since
 * hidden networks need to be explicitly probed for.</li>
 * <li>If not set, defaults to false (i.e not a hidden network).</li>
 *
 * @param isHiddenSsid {@code true} to indicate that the network is hidden, {@code false}
 *                     otherwise.
 * @return Instance of {@link android.net.wifi.WifiNetworkSpecifier.Builder Builder} to enable chaining of the builder method.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.WifiNetworkSpecifier.Builder setIsHiddenSsid(boolean isHiddenSsid) { throw new RuntimeException("Stub!"); }

/**
 * Specifies the preferred channels for this network. The channels set in the request will
 * be used to optimize the scan and connection.
 * <p>
 * <li>Should only be set to request local-only network</li>
 * <li>If not set, defaults to an empty array and device will do a full band scan.</li>
 *
 * @param channelFreqs an Array of the channels in MHz. The length of the array must not
 *                     exceed {@link android.net.wifi.WifiManager#getMaxNumberOfChannelsPerNetworkSpecifierRequest() WifiManager.getMaxNumberOfChannelsPerNetworkSpecifierRequest()}
 *
 * This value cannot be {@code null}.
 * @return Instance of {@link android.net.wifi.WifiNetworkSpecifier.Builder Builder} to enable chaining of the builder method.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.WifiNetworkSpecifier.Builder setPreferredChannelsFrequenciesMhz(@android.annotation.NonNull int[] channelFreqs) { throw new RuntimeException("Stub!"); }

/**
 * Set the unicode SSID for the network.
 * <p>
 * <li>Sets the SSID to use for filtering networks from scan results. Will only match
 * networks whose SSID is identical to the UTF-8 encoding of the specified value.</li>
 * <li>Overrides any previous value set using {@link #setSsid(java.lang.String)} or
 * {@link #setSsidPattern(android.os.PatternMatcher)}.</li>
 *
 * @param ssid The SSID of the network. It must be valid Unicode.
 * This value cannot be {@code null}.
 * @return Instance of {@link android.net.wifi.WifiNetworkSpecifier.Builder Builder} to enable chaining of the builder method.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if the SSID is not valid unicode.
 */

@android.annotation.NonNull
public android.net.wifi.WifiNetworkSpecifier.Builder setSsid(@android.annotation.NonNull java.lang.String ssid) { throw new RuntimeException("Stub!"); }

/**
 * Set the unicode SSID match pattern to use for filtering networks from scan results.
 * <p>
 * <li>Overrides any previous value set using {@link #setSsid(java.lang.String)} or
 * {@link #setSsidPattern(android.os.PatternMatcher)}.</li>
 *
 * @param ssidPattern Instance of {@link android.os.PatternMatcher PatternMatcher} containing the UTF-8 encoded
 *                    string pattern to use for matching the network's SSID.
 * This value cannot be {@code null}.
 * @return Instance of {@link android.net.wifi.WifiNetworkSpecifier.Builder Builder} to enable chaining of the builder method.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.WifiNetworkSpecifier.Builder setSsidPattern(@android.annotation.NonNull android.os.PatternMatcher ssidPattern) { throw new RuntimeException("Stub!"); }

/**
 * Set the associated enterprise configuration for this network. Needed for authenticating
 * to WPA2-EAP networks. See {@link android.net.wifi.WifiEnterpriseConfig WifiEnterpriseConfig} for description. Local-only
 * connection will not support Trust On First Use (TOFU). If TOFU is enabled on this
 * Enterprise Config, framework will reject the connection. See {@link android.net.wifi.WifiEnterpriseConfig#enableTrustOnFirstUse  }
 *
 * @param enterpriseConfig Instance of {@link android.net.wifi.WifiEnterpriseConfig WifiEnterpriseConfig}.
 * This value cannot be {@code null}.
 * @return Instance of {@link android.net.wifi.WifiNetworkSpecifier.Builder Builder} to enable chaining of the builder method.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.WifiNetworkSpecifier.Builder setWpa2EnterpriseConfig(@android.annotation.NonNull android.net.wifi.WifiEnterpriseConfig enterpriseConfig) { throw new RuntimeException("Stub!"); }

/**
 * Set the ASCII WPA2 passphrase for this network. Needed for authenticating to
 * WPA2-PSK networks.
 *
 * @param passphrase passphrase of the network.
 * This value cannot be {@code null}.
 * @return Instance of {@link android.net.wifi.WifiNetworkSpecifier.Builder Builder} to enable chaining of the builder method.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if the passphrase is not ASCII encodable.
 */

@android.annotation.NonNull
public android.net.wifi.WifiNetworkSpecifier.Builder setWpa2Passphrase(@android.annotation.NonNull java.lang.String passphrase) { throw new RuntimeException("Stub!"); }

/**
 * Set the associated enterprise configuration for this network. Needed for authenticating
 * to WPA3-Enterprise in 192-bit security mode networks. See {@link android.net.wifi.WifiEnterpriseConfig WifiEnterpriseConfig}
 * for description. Both the client and CA certificates must be provided, and must be of
 * type of either sha384WithRSAEncryption with key length of 3072bit or more (OID
 * 1.2.840.113549.1.1.12), or ecdsa-with-SHA384 with key length of 384bit or more (OID
 * 1.2.840.10045.4.3.3). Local-only connection will not support Trust On First Use (TOFU).
 * If TOFU is enabled on this Enterprise Config, framework will reject the connection. See
 * {@link android.net.wifi.WifiEnterpriseConfig#enableTrustOnFirstUse WifiEnterpriseConfig.enableTrustOnFirstUse}
 *
 * @param enterpriseConfig Instance of {@link android.net.wifi.WifiEnterpriseConfig WifiEnterpriseConfig}.
 * This value cannot be {@code null}.
 * @return Instance of {@link android.net.wifi.WifiNetworkSpecifier.Builder Builder} to enable chaining of the builder method.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if the EAP type or certificates do not meet 192-bit mode
 *     requirements.
 */

@android.annotation.NonNull
public android.net.wifi.WifiNetworkSpecifier.Builder setWpa3Enterprise192BitModeConfig(@android.annotation.NonNull android.net.wifi.WifiEnterpriseConfig enterpriseConfig) { throw new RuntimeException("Stub!"); }

/**
 * Set the associated enterprise configuration for this network. Needed for authenticating
 * to WPA3-Enterprise networks (standard and 192-bit security). See
 * {@link android.net.wifi.WifiEnterpriseConfig WifiEnterpriseConfig} for description. For 192-bit security networks, both the
 * client and CA certificates must be provided, and must be of type of either
 * sha384WithRSAEncryption (OID 1.2.840.113549.1.1.12) or ecdsa-with-SHA384
 * (OID 1.2.840.10045.4.3.3).
 *
 * @deprecated use {@link #setWpa3EnterpriseStandardModeConfig(android.net.wifi.WifiEnterpriseConfig)} or
 * {@link #setWpa3Enterprise192BitModeConfig(android.net.wifi.WifiEnterpriseConfig)} to specify
 * WPA3-Enterprise type explicitly.
 *
 * @param enterpriseConfig Instance of {@link android.net.wifi.WifiEnterpriseConfig WifiEnterpriseConfig}.
 * This value cannot be {@code null}.
 * @return Instance of {@link android.net.wifi.WifiNetworkSpecifier.Builder Builder} to enable chaining of the builder method.
 
 * This value cannot be {@code null}.
 */

@Deprecated
@android.annotation.NonNull
public android.net.wifi.WifiNetworkSpecifier.Builder setWpa3EnterpriseConfig(@android.annotation.NonNull android.net.wifi.WifiEnterpriseConfig enterpriseConfig) { throw new RuntimeException("Stub!"); }

/**
 * Set the associated enterprise configuration for this network. Needed for authenticating
 * to standard WPA3-Enterprise networks. See {@link android.net.wifi.WifiEnterpriseConfig WifiEnterpriseConfig} for description.
 * For WPA3-Enterprise in 192-bit security mode networks, see {@link
 * #setWpa3Enterprise192BitModeConfig(android.net.wifi.WifiEnterpriseConfig)} for description. Local-only
 * connection will not support Trust On First Use (TOFU). If TOFU is enabled on this
 * Enterprise Config, framework will reject the connection. See {@link android.net.wifi.WifiEnterpriseConfig#enableTrustOnFirstUse  }
 *
 * @param enterpriseConfig Instance of {@link android.net.wifi.WifiEnterpriseConfig WifiEnterpriseConfig}.
 * This value cannot be {@code null}.
 * @return Instance of {@link android.net.wifi.WifiNetworkSpecifier.Builder Builder} to enable chaining of the builder method.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.WifiNetworkSpecifier.Builder setWpa3EnterpriseStandardModeConfig(@android.annotation.NonNull android.net.wifi.WifiEnterpriseConfig enterpriseConfig) { throw new RuntimeException("Stub!"); }

/**
 * Set the ASCII WPA3 passphrase for this network. Needed for authenticating to WPA3-SAE
 * networks.
 *
 * @param passphrase passphrase of the network.
 * This value cannot be {@code null}.
 * @return Instance of {@link android.net.wifi.WifiNetworkSpecifier.Builder Builder} to enable chaining of the builder method.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if the passphrase is not ASCII encodable.
 */

@android.annotation.NonNull
public android.net.wifi.WifiNetworkSpecifier.Builder setWpa3Passphrase(@android.annotation.NonNull java.lang.String passphrase) { throw new RuntimeException("Stub!"); }
}

}

