/*
 * Copyright (C) 2019 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.net.ipsec.ike;

import android.net.ipsec.ike.exceptions.IkeException;

/**
 * Callback interface for receiving state changes of an {@link android.net.ipsec.ike.IkeSession IkeSession}.
 *
 * <p>{@link android.net.ipsec.ike.IkeSessionCallback IkeSessionCallback} MUST be unique to each {@link android.net.ipsec.ike.IkeSession IkeSession}. It is registered when
 * callers are requesting a new {@link android.net.ipsec.ike.IkeSession IkeSession}. It is automatically unregistered when an {@link android.net.ipsec.ike.IkeSession IkeSession} is closed.
 *
 * @see <a href="https://tools.ietf.org/html/rfc7296">RFC 7296, Internet Key Exchange Protocol
 *     Version 2 (IKEv2)</a>
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public interface IkeSessionCallback {

/**
 * Called when the {@link android.net.ipsec.ike.IkeSession IkeSession} is closed.
 *
 * <p>When the closure is caused by a local, fatal error, {@link
 * #onClosedWithException(android.net.ipsec.ike.exceptions.IkeException)} will be fired instead of this method.
 */

public void onClosed();

/**
 * Called if {@link android.net.ipsec.ike.IkeSession IkeSession} setup failed or {@link android.net.ipsec.ike.IkeSession IkeSession} is closed because of a fatal
 * error.
 *
 * @param exception the detailed error information.

 * This value cannot be {@code null}.
 */

public default void onClosedWithException(@android.annotation.NonNull android.net.ipsec.ike.exceptions.IkeException exception) { throw new RuntimeException("Stub!"); }

/**
 * Called if a recoverable error is encountered in an established {@link android.net.ipsec.ike.IkeSession IkeSession}.
 *
 * <p>This method may be triggered by protocol errors such as an INVALID_IKE_SPI, or by
 * non-protocol errors such as the underlying {@link android.net.Network} dying.
 *
 * @param exception the detailed error information.

 * This value cannot be {@code null}.
 */

public default void onError(@android.annotation.NonNull android.net.ipsec.ike.exceptions.IkeException exception) { throw new RuntimeException("Stub!"); }

/**
 * Called when the {@link android.net.ipsec.ike.IkeSession IkeSession} setup succeeds.
 *
 * <p>This method does not indicate the first Child Session has been setup. Caller MUST refer to
 * the corresponding {@link android.net.ipsec.ike.ChildSessionCallback ChildSessionCallback} for the Child Session setup result.
 *
 * @param sessionConfiguration the configuration information of {@link android.net.ipsec.ike.IkeSession IkeSession} negotiated
 *     during IKE setup.

 * This value cannot be {@code null}.
 */

public void onOpened(@android.annotation.NonNull android.net.ipsec.ike.IkeSessionConfiguration sessionConfiguration);
}

