/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.health.connect;

import android.content.Context;
import android.health.connect.datatypes.AggregationType;
import android.health.connect.datatypes.FhirResource;
import android.health.connect.datatypes.FhirVersion;
import android.health.connect.datatypes.MedicalDataSource;
import android.health.connect.datatypes.MedicalResource;
import android.health.connect.datatypes.Record;
import android.os.OutcomeReceiver;
import java.time.Duration;
import java.time.Period;
import java.time.ZoneOffset;
import java.util.List;
import java.util.concurrent.Executor;

/**
 * This class provides APIs to interact with the centralized HealthConnect storage maintained by the
 * system.
 *
 * <p>HealthConnect is an offline, on-device storage that unifies data from multiple devices and
 * apps into an ecosystem featuring.
 *
 * <ul>
 *   <li>APIs to insert data of various types into the system.
 * </ul>
 *
 * <p>The basic unit of data in HealthConnect is represented as a {@link android.health.connect.datatypes.Record Record} object, which is
 * the base class for all the other data types such as {@link
 * android.health.connect.datatypes.StepsRecord}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class HealthConnectManager {

HealthConnectManager() { throw new RuntimeException("Stub!"); }

/**
 * Get aggregations corresponding to {@code request}.
 *
 * @param <T> Result type of the aggregation.
 *     <p>Note:
 *     <p>This type is embedded in the {@link android.health.connect.datatypes.AggregationType AggregationType} as {@link android.health.connect.datatypes.AggregationType AggregationType} are
 *     typed in nature.
 *     <p>Only {@link android.health.connect.datatypes.AggregationType AggregationType}s that are of same type T can be queried together
 * @param request request for different aggregation.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @see android.health.connect.AggregateRecordsResponse#get
 */

public <T> void aggregate(@android.annotation.NonNull android.health.connect.AggregateRecordsRequest<T> request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<android.health.connect.AggregateRecordsResponse<T>,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Get aggregations corresponding to {@code request}. Use this API if results are to be grouped
 * by concrete intervals of time, for example 5 Hrs, 10 Hrs etc.
 *
 * @param <T> Result type of the aggregation.
 *     <p>Note:
 *     <p>This type is embedded in the {@link android.health.connect.datatypes.AggregationType AggregationType} as {@link android.health.connect.datatypes.AggregationType AggregationType} are
 *     typed in nature.
 *     <p>Only {@link android.health.connect.datatypes.AggregationType AggregationType}s that are of same type T can be queried together
 * @param request request for different aggregation.
 * This value cannot be {@code null}.
 * @param duration Duration on which to group by results
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @see android.health.connect.HealthConnectManager#aggregateGroupByPeriod
 */

public <T> void aggregateGroupByDuration(@android.annotation.NonNull android.health.connect.AggregateRecordsRequest<T> request, @android.annotation.NonNull java.time.Duration duration, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.util.List<android.health.connect.AggregateRecordsGroupedByDurationResponse<T>>,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Get aggregations corresponding to {@code request}. Use this API if results are to be grouped
 * by number of days. This API handles changes in {@link java.time.ZoneOffset ZoneOffset} when computing the data on
 * a per-day basis.
 *
 * @param <T> Result type of the aggregation.
 *     <p>Note:
 *     <p>This type is embedded in the {@link android.health.connect.datatypes.AggregationType AggregationType} as {@link android.health.connect.datatypes.AggregationType AggregationType} are
 *     typed in nature.
 *     <p>Only {@link android.health.connect.datatypes.AggregationType AggregationType}s that are of same type T can be queried together
 * @param request Request for different aggregation.
 * This value cannot be {@code null}.
 * @param period Period on which to group by results
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @see android.health.connect.AggregateRecordsGroupedByPeriodResponse#get
 * @see android.health.connect.HealthConnectManager#aggregateGroupByDuration
 */

public <T> void aggregateGroupByPeriod(@android.annotation.NonNull android.health.connect.AggregateRecordsRequest<T> request, @android.annotation.NonNull java.time.Period period, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.util.List<android.health.connect.AggregateRecordsGroupedByPeriodResponse<T>>,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Creates a {@link android.health.connect.datatypes.MedicalDataSource MedicalDataSource} in HealthConnect based on the {@link android.health.connect.CreateMedicalDataSourceRequest CreateMedicalDataSourceRequest} request values.
 *
 * <p>Medical data is represented using the <a href="https://hl7.org/fhir/">Fast Healthcare
 * Interoperability Resources (FHIR)</a> standard.
 *
 * <p>A {@link android.health.connect.datatypes.MedicalDataSource MedicalDataSource} needs to be created before any {@link android.health.connect.datatypes.MedicalResource MedicalResource}s for
 * that source can be inserted. Separate {@link android.health.connect.datatypes.MedicalDataSource MedicalDataSource}s should be created for
 * medical records coming from different sources (e.g. different FHIR endpoints, different
 * healthcare systems), unless the data has been reconciled and all records have a unique
 * combination of resource type and resource id.
 *
 * <p>The {@link android.health.connect.CreateMedicalDataSourceRequest.Builder#setDisplayName display name} must be
 * unique per app, and {@link android.health.connect.CreateMedicalDataSourceRequest.Builder#setFhirVersion CreateMedicalDataSourceRequest.Builder.setFhirVersion} FHIR
 * version} must be a version supported by Health Connect, as documented on the {@link android.health.connect.datatypes.FhirVersion FhirVersion}. See {@link android.health.connect.CreateMedicalDataSourceRequest.Builder#setFhirBaseUri CreateMedicalDataSourceRequest.Builder.setFhirBaseUri} for more
 * details on the FHIR base URI.
 *
 * <br>
 * Requires {@link android.health.connect.HealthPermissions#WRITE_MEDICAL_DATA}
 * @param request Creation request.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if {@code request} contains a FHIR base URI or display name
 *     exceeding the character limits, or an unsupported FHIR version.
 */

public void createMedicalDataSource(@android.annotation.NonNull android.health.connect.CreateMedicalDataSourceRequest request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<android.health.connect.datatypes.MedicalDataSource,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Deletes a {@link android.health.connect.datatypes.MedicalDataSource MedicalDataSource} and all data linked to it.
 *
 * <p>If the provided data source {@code id} is either invalid, or does not exist, or owned by
 * another apps, the caller will receive an exception with code {@link android.health.connect.HealthConnectException#ERROR_INVALID_ARGUMENT  } via {@code callback.onError()}.
 *
 * <p>Regarding permissions:
 *
 * <ul>
 *   <li>Only apps with the system permission can delete data written by apps other than
 *       themselves.
 *   <li>Deletes are permitted in the foreground or background.
 * </ul>
 *
 * <br>
 * Requires {@link android.health.connect.HealthPermissions#WRITE_MEDICAL_DATA} or android.health.connect.HealthPermissions.MANAGE_HEALTH_DATA_PERMISSION
 * @param id The id of the data source to delete.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * @param callback Callback to receive result of performing this operation.

 * This value cannot be {@code null}.
 */

public void deleteMedicalDataSourceWithData(@android.annotation.NonNull java.lang.String id, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.lang.Void,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Deletes {@link android.health.connect.datatypes.MedicalResource MedicalResource}s based on given filters in {@link android.health.connect.DeleteMedicalResourcesRequest DeleteMedicalResourcesRequest}.
 *
 * <p>Regarding permissions:
 *
 * <ul>
 *   <li>Only apps with the system permission can delete data written by apps other than
 *       themselves.
 *   <li>Deletes are permitted in the foreground or background.
 * </ul>
 *
 * <br>
 * Requires {@link android.health.connect.HealthPermissions#WRITE_MEDICAL_DATA} or android.health.connect.HealthPermissions.MANAGE_HEALTH_DATA_PERMISSION
 * @param request The delete request.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if {@code request} contains unsupported medical resource
 *     types or invalid {@link android.health.connect.datatypes.MedicalDataSource MedicalDataSource} IDs.
 */

public void deleteMedicalResources(@android.annotation.NonNull android.health.connect.DeleteMedicalResourcesRequest request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.lang.Void,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Deletes a list of {@link android.health.connect.datatypes.MedicalResource MedicalResource}s by the provided list of {@link android.health.connect.MedicalResourceId MedicalResourceId}s.
 *
 * <ul>
 *   <li>If any ID in {@code ids} is invalid, the API will throw an {@link java.lang.IllegalArgumentException IllegalArgumentException}, and nothing will be deleted.
 *   <li>If any ID in {@code ids} does not exist, that ID will be ignored, while deletion on
 *       other IDs will be performed.
 * </ul>
 *
 * <p>Regarding permissions:
 *
 * <ul>
 *   <li>Only apps with the system permission can delete data written by apps other than
 *       themselves.
 *   <li>Deletes are permitted in the foreground or background.
 * </ul>
 *
 * <br>
 * Requires {@link android.health.connect.HealthPermissions#WRITE_MEDICAL_DATA} or android.health.connect.HealthPermissions.MANAGE_HEALTH_DATA_PERMISSION
 * @param ids The ids to delete.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * @param callback Callback to receive result of performing this operation.

 * This value cannot be {@code null}.
 */

public void deleteMedicalResources(@android.annotation.NonNull java.util.List<android.health.connect.MedicalResourceId> ids, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.lang.Void,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Deletes records based on the {@link android.health.connect.TimeRangeFilter TimeRangeFilter}.
 *
 * <p>Deletions are performed in a transaction i.e. either all will be deleted or none
 *
 * @param recordType recordType to perform delete operation on.
 * This value cannot be {@code null}.
 * @param timeRangeFilter time filter based on which to delete the records.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * @param callback Callback to receive result of performing this operation.

 * This value cannot be {@code null}.
 */

public void deleteRecords(@android.annotation.NonNull java.lang.Class<? extends android.health.connect.datatypes.Record> recordType, @android.annotation.NonNull android.health.connect.TimeRangeFilter timeRangeFilter, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.lang.Void,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Deletes records based on {@link android.health.connect.RecordIdFilter RecordIdFilter}.
 *
 * <p>Deletions are performed in a transaction i.e. either all will be deleted or none
 *
 * @param recordIds recordIds on which to perform delete operation.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if {@code recordIds is empty}
 */

public void deleteRecords(@android.annotation.NonNull java.util.List<android.health.connect.RecordIdFilter> recordIds, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.lang.Void,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Get token for {HealthConnectManager#getChangeLogs}. Changelogs requested corresponding to
 * this token will be post the time this token was generated by the system all items that match
 * the given filters.
 *
 * <p>Tokens from this request are to be passed to {HealthConnectManager#getChangeLogs}
 *
 * @param request A request to get changelog token
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * @param callback Callback to receive result of performing this operation.

 * This value cannot be {@code null}.
 */

public void getChangeLogToken(@android.annotation.NonNull android.health.connect.changelog.ChangeLogTokenRequest request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<android.health.connect.changelog.ChangeLogTokenResponse,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Get change logs post the time when {@code token} was generated.
 *
 * @param changeLogsRequest The token from {@link android.health.connect.HealthConnectManager#getChangeLogToken HealthConnectManager.getChangeLogToken}.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @see android.health.connect.HealthConnectManager#getChangeLogToken
 */

public void getChangeLogs(@android.annotation.NonNull android.health.connect.changelog.ChangeLogsRequest changeLogsRequest, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<android.health.connect.changelog.ChangeLogsResponse,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Returns the requested {@link android.health.connect.datatypes.MedicalDataSource MedicalDataSource}s.
 *
 * <p>Number of data sources returned by this API will depend based on below factors:
 *
 * <ul>
 *   <li>If an empty {@link android.health.connect.GetMedicalDataSourcesRequest#getPackageNames()        names} is passed, all permitted data sources from all apps will be returned. See below.
 *   <li>If any package name in the {@link android.health.connect.GetMedicalDataSourcesRequest#getPackageNames()        of package names} is invalid, the API will throw an {@link java.lang.IllegalArgumentException IllegalArgumentException}.
 *   <li>If a non-empty {@link android.health.connect.GetMedicalDataSourcesRequest#getPackageNames()        names} is specified in the request, then only the permitted data sources created by
 *       those packages will be returned. See below.
 * </ul>
 *
 * <p>There is no specific read permission for getting data sources. Instead permission to read
 * data sources is based on whether the caller has permission to read the data currently linked
 * to that data source. Being permitted to get data sources is dependent on the following logic,
 * in priority order, earlier statements take precedence.
 *
 * <ol>
 *   <li>A caller with the system permission can get any data source in the foreground or
 *       background.
 *   <li>A caller without any read or write permissions for health data will not be able to get
 *       any medical data sources and receive an exception with code {@link android.health.connect.HealthConnectException#ERROR_SECURITY        } via {@code callback.onError()}, even for data
 *       sources the caller has created.
 *   <li>Callers can get data sources they have created, whether this method is called in the
 *       foreground or background. Note this only applies if the caller has at least one read or
 *       write permission for health data.
 *   <li>For any given data source, a caller can get that data source in the foreground if the
 *       caller has permission to read any of the data linked to that data source. For clarity,
 *       the does not allow it to get an empty data source.
 *   <li>For any given data source, a caller can get that data source in the background if it
 *       has both permission to read any of the data linked to that data source, and {@link
 *       android.health.connect.HealthPermissions#READ_HEALTH_DATA_IN_BACKGROUND}.
 *   <li>In all other cases the caller is not permitted to get the given data source and it will
 *       not be returned.
 * </ol>
 *
 * @param request the request for which data sources to return.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if {@code request} contains invalid package names.
 */

public void getMedicalDataSources(@android.annotation.NonNull android.health.connect.GetMedicalDataSourcesRequest request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.util.List<android.health.connect.datatypes.MedicalDataSource>,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Returns {@link android.health.connect.datatypes.MedicalDataSource MedicalDataSource}s for the provided list of IDs.
 *
 * <p>The number and order of medical data sources returned by this API is not guaranteed. The
 * number will depend on the factors below:
 *
 * <ul>
 *   <li>If an empty list of {@code ids} is provided, an empty list will be returned.
 *   <li>If any ID in {@code ids} is invalid, the caller will receive an exception with code
 *       {@link android.health.connect.HealthConnectException#ERROR_INVALID_ARGUMENT HealthConnectException.ERROR_INVALID_ARGUMENT} via {@code callback.onError()}.
 *   <li>If any ID in {@code ids} does not exist, no data source will be returned for that ID.
 *   <li>Callers will only get data sources they are permitted to get. See below.
 * </ul>
 *
 * <p>There is no specific read permission for getting data sources. Instead, permission to read
 * data sources is based on whether the caller has permission to read the data currently linked
 * to that data source. Being permitted to get data sources is dependent on the following logic,
 * in priority order, earlier statements take precedence.
 *
 * <ol>
 *   <li>A caller with the system permission can get any data source in the foreground or
 *       background.
 *   <li>A caller without any read or write permissions for health data will not be able to get
 *       any medical data sources and receive an exception with code {@link android.health.connect.HealthConnectException#ERROR_SECURITY        } via {@code callback.onError()}, even for data
 *       sources the caller has created.
 *   <li>Callers can get data sources they have created, whether this method is called in the
 *       foreground or background. Note this only applies if the caller has at least one read or
 *       write permission for health data.
 *   <li>For any given data source, a caller can get that data source in the foreground if the
 *       caller has permission to read any of the data linked to that data source. For clarity,
 *       the does not allow it to get an empty data source.
 *   <li>For any given data source, a caller can get that data source in the background if it
 *       has both permission to read any of the data linked to that data source, and {@link
 *       android.health.connect.HealthPermissions#READ_HEALTH_DATA_IN_BACKGROUND}.
 *   <li>In all other cases the caller is not permitted to get the given data source and it will
 *       not be returned.
 * </ol>
 *
 * @param ids Identifiers for data sources to get.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if the size of {@code ids} is more than 5000.
 */

public void getMedicalDataSources(@android.annotation.NonNull java.util.List<java.lang.String> ids, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.util.List<android.health.connect.datatypes.MedicalDataSource>,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Inserts {@code records} into the HealthConnect database. The records returned in {@link android.health.connect.InsertRecordsResponse InsertRecordsResponse} contains the unique IDs of the input records. The values are in same
 * order as {@code records}. In case of an error or a permission failure the HealthConnect
 * service, {@link android.os.OutcomeReceiver#onError OutcomeReceiver.onError} will be invoked with a {@link android.health.connect.HealthConnectException HealthConnectException}.
 *
 * @param records list of records to be inserted.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @throws java.lang.RuntimeException for internal errors
 */

public void insertRecords(@android.annotation.NonNull java.util.List<android.health.connect.datatypes.Record> records, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<android.health.connect.InsertRecordsResponse,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Reads {@link android.health.connect.datatypes.MedicalResource MedicalResource}s based on {@link android.health.connect.ReadMedicalResourcesInitialRequest ReadMedicalResourcesInitialRequest} or {@link android.health.connect.ReadMedicalResourcesPageRequest ReadMedicalResourcesPageRequest}.
 *
 * <p>Being permitted to read medical resources is dependent on the following logic, in priority
 * order, earlier statements take precedence.
 *
 * <ol>
 *   <li>A caller with the system permission can get any medical resources in the foreground or
 *       background.
 *   <li>A caller without any read or write permissions for health data will not be able to get
 *       any medical resources and receive an exception with code {@link android.health.connect.HealthConnectException#ERROR_SECURITY        } via {@code callback.onError()}, even for medical
 *       resources the caller has created.
 *   <li>Callers can get medical resources they have created, whether this method is called in
 *       the foreground or background. Note this only applies if the caller has at least one
 *       read or write permission for health data.
 *   <li>For any given medical resource, a caller can get that medical resource in the
 *       foreground if the caller has the corresponding read permission, or in the background if
 *       it also has {@link
 *       android.health.connect.HealthPermissions#READ_HEALTH_DATA_IN_BACKGROUND}.
 *   <li>In all other cases the caller is not permitted to get the given medical resource and it
 *       will not be returned.
 * </ol>
 *
 * @param request The read request {@link android.health.connect.ReadMedicalResourcesInitialRequest ReadMedicalResourcesInitialRequest} or {@link android.health.connect.ReadMedicalResourcesPageRequest ReadMedicalResourcesPageRequest}.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if {@code request} has set page size to be less than 1 or
 *     more than 5000; or if contains unsupported medical resource type or invalid {@link android.health.connect.datatypes.MedicalDataSource MedicalDataSource} IDs when using {@link android.health.connect.ReadMedicalResourcesInitialRequest ReadMedicalResourcesInitialRequest}.
 */

public void readMedicalResources(@android.annotation.NonNull android.health.connect.ReadMedicalResourcesRequest request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<android.health.connect.ReadMedicalResourcesResponse,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Reads {@link android.health.connect.datatypes.MedicalResource MedicalResource}s based on a list of {@link android.health.connect.MedicalResourceId MedicalResourceId}s.
 *
 * <p>The number and order of medical resources returned by this API is not guaranteed. The
 * number will depend on the factors below:
 *
 * <ul>
 *   <li>If an empty list of {@code ids} is provided, an empty list will be returned.
 *   <li>If the size of {@code ids} is more than 5000, the API will throw an {@link java.lang.IllegalArgumentException IllegalArgumentException}.
 *   <li>If any ID in {@code ids} is invalid, the API will throw an {@link java.lang.IllegalArgumentException IllegalArgumentException}.
 *   <li>If any ID in {@code ids} does not exist, no medical resource will be returned for that
 *       ID.
 *   <li>Callers will only get medical resources they are permitted to get. See below.
 * </ul>
 *
 * Being permitted to read medical resources is dependent on the following logic, in priority
 * order, earlier statements take precedence.
 *
 * <ol>
 *   <li>A caller with the system permission can get any medical resources in the foreground or
 *       background.
 *   <li>A caller without any read or write permissions for health data will not be able to get
 *       any medical resources and receive an exception with code {@link android.health.connect.HealthConnectException#ERROR_SECURITY        } via {@code callback.onError()}, even for medical
 *       resources the caller has created.
 *   <li>Callers can get medical resources they have created, whether this method is called in
 *       the foreground or background. Note this only applies if the caller has at least one
 *       read or write permission for health data.
 *   <li>For any given medical resource, a caller can get that medical resource in the
 *       foreground if the caller has the corresponding read permission, or in the background if
 *       it also has {@link
 *       android.health.connect.HealthPermissions#READ_HEALTH_DATA_IN_BACKGROUND}.
 *   <li>In all other cases the caller is not permitted to get the given medical resource and it
 *       will not be returned.
 * </ol>
 *
 * <p>Each returned {@link android.health.connect.datatypes.MedicalResource MedicalResource} has passed the Health Connect FHIR validation checks
 * at write time, but is not guaranteed to meet all requirements of the <a
 * href="https://hl7.org/fhir/resourcelist.html">Fast Healthcare Interoperability Resources
 * (FHIR) spec</a>. If required, clients should perform their own checks on the data.
 *
 * @param ids Identifiers on which to perform read operation.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if the size of {@code ids} is more than 5000 or if any id is
 *     invalid.
 */

public void readMedicalResources(@android.annotation.NonNull java.util.List<android.health.connect.MedicalResourceId> ids, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.util.List<android.health.connect.datatypes.MedicalResource>,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * API to read records based on {@link android.health.connect.ReadRecordsRequestUsingFilters ReadRecordsRequestUsingFilters} or {@link android.health.connect.ReadRecordsRequestUsingIds ReadRecordsRequestUsingIds}
 *
 * <p>Number of records returned by this API will depend based on below factors:
 *
 * <p>When an app with read permission allowed calls the API from background then it will be
 * able to read only its own inserted records and will not get records inserted by other apps.
 * This may be less than the total records present for the record type.
 *
 * <p>When an app with read permission allowed calls the API from foreground then it will be
 * able to read all records for the record type.
 *
 * <p>App with only write permission but no read permission allowed will be able to read only
 * its own inserted records both when in foreground or background.
 *
 * <p>An app without both read and write permissions will not be able to read any record and the
 * API will throw Security Exception.
 *
 * @param request Read request based on {@link android.health.connect.ReadRecordsRequestUsingFilters ReadRecordsRequestUsingFilters} or {@link android.health.connect.ReadRecordsRequestUsingIds ReadRecordsRequestUsingIds}
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if request page size set is more than 5000 in {@link android.health.connect.ReadRecordsRequestUsingFilters ReadRecordsRequestUsingFilters}
 * @throws java.lang.SecurityException if app without read or write permission tries to read.
 */

public <T extends android.health.connect.datatypes.Record> void readRecords(@android.annotation.NonNull android.health.connect.ReadRecordsRequest<T> request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<android.health.connect.ReadRecordsResponse<T>,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Updates {@code records} into the HealthConnect database. In case of an error or a permission
 * failure the HealthConnect service, {@link android.os.OutcomeReceiver#onError OutcomeReceiver.onError} will be invoked with a
 * {@link android.health.connect.HealthConnectException HealthConnectException}.
 *
 * <p>In case the input record to be updated does not exist in the database or the caller is not
 * the owner of the record then {@link android.health.connect.HealthConnectException#ERROR_INVALID_ARGUMENT HealthConnectException.ERROR_INVALID_ARGUMENT} will be
 * thrown.
 *
 * @param records list of records to be updated.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if at least one of the records is missing both
 *     ClientRecordID and UUID.
 */

public void updateRecords(@android.annotation.NonNull java.util.List<android.health.connect.datatypes.Record> records, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.lang.Void,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Inserts or updates a list of {@link android.health.connect.datatypes.MedicalResource MedicalResource}s into the HealthConnect database using
 * {@link android.health.connect.UpsertMedicalResourceRequest UpsertMedicalResourceRequest}.
 *
 * <p>For each {@link android.health.connect.UpsertMedicalResourceRequest UpsertMedicalResourceRequest}, one {@link android.health.connect.datatypes.MedicalResource MedicalResource} will be
 * returned. The returned list of {@link android.health.connect.datatypes.MedicalResource MedicalResource}s will be in the same order as the
 * {@code requests}.
 *
 * <p>Note that a {@link android.health.connect.datatypes.MedicalDataSource MedicalDataSource} needs to be created using {@link
 * #createMedicalDataSource} before any {@link android.health.connect.datatypes.MedicalResource MedicalResource}s can be upserted for this
 * source.
 *
 * <p>Medical data is represented using the <a href="https://hl7.org/fhir/">Fast Healthcare
 * Interoperability Resources (FHIR)</a> standard. The FHIR resource provided in {@link android.health.connect.UpsertMedicalResourceRequest#getData()  } is expected to be valid FHIR in JSON representation
 * for the specified {@link android.health.connect.UpsertMedicalResourceRequest#getFhirVersion() UpsertMedicalResourceRequest.getFhirVersion()} according to the <a
 * href="https://hl7.org/fhir/resourcelist.html">FHIR spec</a>. Structural validation checks
 * such as resource structure, field types and presence of required fields are performed, but
 * these checks may not cover all FHIR spec requirements and may change in future versions.
 *
 * <p>Data written to Health Connect should be for a single individual only. However, the API
 * allows for multiple Patient resources to be written to account for the possibility of
 * multiple Patient resources being present in one individual's medical record.
 *
 * <p>Each {@link android.health.connect.UpsertMedicalResourceRequest UpsertMedicalResourceRequest} also has to meet the following requirements.
 *
 * <ul>
 *   <li>The FHIR resource contains an "id" and "resourceType" field.
 *   <li>The FHIR resource type is in our accepted list of resource types. See {@link android.health.connect.datatypes.FhirResource FhirResource} for the accepted types.
 *   <li>The FHIR resource does not contain any "contained" resources.
 *   <li>The resource can be mapped to one of the READ_MEDICAL_DATA_ {@link android.health.connect.HealthPermissions HealthPermissions}
 *       categories.
 *   <li>The {@link android.health.connect.UpsertMedicalResourceRequest#getDataSourceId() UpsertMedicalResourceRequest.getDataSourceId()} is valid.
 *   <li>The {@link android.health.connect.UpsertMedicalResourceRequest#getFhirVersion() UpsertMedicalResourceRequest.getFhirVersion()} matches the {@link android.health.connect.datatypes.FhirVersion FhirVersion} of the {@link android.health.connect.datatypes.MedicalDataSource MedicalDataSource}.
 * </ul>
 *
 * <p>If any request contains invalid {@link android.health.connect.datatypes.MedicalDataSource MedicalDataSource} IDs, the API will throw an
 * {@link java.lang.IllegalArgumentException IllegalArgumentException}, and none of the {@code requests} will be upserted into the
 * HealthConnect database.
 *
 * <p>If any request is deemed invalid for any other reasons, the caller will receive an
 * exception with code {@link android.health.connect.HealthConnectException#ERROR_INVALID_ARGUMENT HealthConnectException.ERROR_INVALID_ARGUMENT} via {@code
 * callback.onError()}, and none of the {@code requests} will be upserted into the HealthConnect
 * database.
 *
 * <p>If data for any {@link android.health.connect.UpsertMedicalResourceRequest UpsertMedicalResourceRequest} fails to be upserted, then no data
 * from any {@code requests} will be upserted into the database.
 *
 * <p>The uniqueness of each request is calculated comparing the combination of {@link android.health.connect.UpsertMedicalResourceRequest#getDataSourceId() data source id}, FHIR resource type and FHIR
 * resource ID extracted from the provided {@link android.health.connect.UpsertMedicalResourceRequest#getData() data}.
 * If the above combination does not match with an existing one in Health Connect, then a new
 * {@link android.health.connect.datatypes.MedicalResource MedicalResource} is inserted, otherwise the existing one is updated.
 *
 * <br>
 * Requires {@link android.health.connect.HealthPermissions#WRITE_MEDICAL_DATA}
 * @param requests List of upsert requests.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive result of performing this operation.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if any {@code requests} contains invalid {@link android.health.connect.datatypes.MedicalDataSource MedicalDataSource} IDs.
 */

public void upsertMedicalResources(@android.annotation.NonNull java.util.List<android.health.connect.UpsertMedicalResourceRequest> requests, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<java.util.List<android.health.connect.datatypes.MedicalResource>,android.health.connect.HealthConnectException> callback) { throw new RuntimeException("Stub!"); }

/**
 * Activity action: Launch UI to manage (e.g.&nbsp;grant/revoke) health permissions.
 *
 * <p>Shows a list of apps which request at least one permission of the Health permission group.
 *
 * <p>Input: {@link android.content.Intent#EXTRA_PACKAGE_NAME} string extra with the name of the
 * app requesting the action. Optional: Adding package name extras launches a UI to manager
 * (e.g. grant/revoke) for this app.
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.ACTIVITY_INTENT_ACTION) public static final java.lang.String ACTION_MANAGE_HEALTH_PERMISSIONS = "android.health.connect.action.MANAGE_HEALTH_PERMISSIONS";

/**
 * Activity action: Launch UI to share the route associated with an exercise session.
 *
 * <p>Input: caller must provide `String` extra EXTRA_SESSION_ID
 *
 * <p>Result will be delivered via [Activity.onActivityResult] with `ExerciseRoute`
 * EXTRA_EXERCISE_ROUTE.
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.ACTIVITY_INTENT_ACTION) public static final java.lang.String ACTION_REQUEST_EXERCISE_ROUTE = "android.health.connect.action.REQUEST_EXERCISE_ROUTE";

/**
 * Used in conjunction with {@link android.content.Intent#ACTION_VIEW_PERMISSION_USAGE} to
 * launch UI to show an app’s health permission rationale/data policy.
 *
 * <p><b>Note:</b> Used by apps to define an intent filter in conjunction with {@link
 * android.content.Intent#ACTION_VIEW_PERMISSION_USAGE} that the HC UI can link out to.
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.INTENT_CATEGORY) public static final java.lang.String CATEGORY_HEALTH_PERMISSIONS = "android.intent.category.HEALTH_PERMISSIONS";

/**
 * An exercise route requested via {@link #ACTION_REQUEST_EXERCISE_ROUTE}.
 *
 * <p>This is returned for a successful request to access a route associated with an exercise
 * session.
 */

public static final java.lang.String EXTRA_EXERCISE_ROUTE = "android.health.connect.extra.EXERCISE_ROUTE";

/**
 * A string ID of a session to be used with {@link #ACTION_REQUEST_EXERCISE_ROUTE}.
 *
 * <p>This is used to specify route of which exercise session we want to request.
 */

public static final java.lang.String EXTRA_SESSION_ID = "android.health.connect.extra.SESSION_ID";
}

